/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.views.swt;

import org.eclipse.jface.viewers.ISelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.simantics.Simantics;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.VirtualGraph;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.common.request.WriteResultRequest;
import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ServiceNotFoundException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.request.Read;
import org.simantics.layer0.Layer0;
import org.simantics.scenegraph.ontology.ScenegraphResources;
import org.simantics.scl.runtime.function.Function3;
import org.simantics.ui.workbench.ResourceEditorPart2;
import org.simantics.utils.ui.jface.ActiveSelectionProvider;
import org.simantics.views.swt.client.base.SWTRoot;

/**
 * To use this class, first model your view contents in .pgraph files according
 * to the Browsing.pgraph ontology. After that there are two ways to put your
 * configuration to use by defining a new view extension:
 * <ol>
 * <li>Set view extension class to
 * <code>org.simantics.browsing.ui.swt.ModelledView:configurationURI=ConfigURI</code>
 * , where ConfigURI is the URI of your view configuration.</li>
 * <li>Extend this class and override at least {@link #configurationURI()} to
 * define the URI from which the configuration for the view is found. Set view
 * extension class to the created class.</li>
 * </ol>
 * 
 * @author Antti Villberg
 */
abstract public class ModelledEditor extends ResourceEditorPart2 {

	public SWTRoot root;
    private Composite base;
	
    abstract protected String configurationURI();
	
    @Override
    public void createPartControl(Composite parent) {
    	
        try {
        	
        	final Variable variable = getResourceInput2().getVariable();
        	
            Resource runtime = Simantics.getSession().sync(new WriteResultRequest<Resource>(Simantics.getSession().getService(VirtualGraph.class)) {
                @Override
                public Resource perform(WriteGraph graph) throws DatabaseException {
                	Layer0 L0 = Layer0.getInstance(graph);
                	ScenegraphResources SG = ScenegraphResources.getInstance(graph);
                	Resource resource = graph.newResource();
					graph.claim(resource, L0.InstanceOf, null, SG.Runtime);
                	graph.claimLiteral(resource, SG.Runtime_HasVariable, variable.getURI(graph), Bindings.STRING);
                	return resource;
                }
            });
            
            base = new Composite(parent, SWT.NONE);
            base.setLayout(new FillLayout());
            
			SWTViewLoaderProcess loader = new SWTViewLoaderProcess(null, null);
            final Variable editorVariable = loader.getVariable(getSession(), configurationURI(), runtime);

        	final Function3<WriteGraph, Variable, Variable, Boolean> onLoaded = getSession().syncRequest(new Read<Function3<WriteGraph, Variable, Variable, Boolean>>() {

        		@Override
        		public Function3<WriteGraph, Variable, Variable, Boolean> perform(ReadGraph graph) throws DatabaseException {
        			return editorVariable.getPossiblePropertyValue(graph, "onLoaded");
        		}

        	});

        	if(onLoaded != null) {

        		Simantics.getSession().sync(new WriteRequest() {
                    @Override
                    public void perform(WriteGraph graph) throws DatabaseException {
                    	onLoaded.apply(graph, editorVariable, variable);
                    }
                });

        	}
            
			root = loader.load(getSession(), editorVariable);
			root.createControls(base);
            
        } catch (ServiceNotFoundException e) {
            Logger.defaultLogError(e);
        } catch (DatabaseException e) {
            Logger.defaultLogError(e);
        }
        
        getSite().setSelectionProvider(new ActiveSelectionProvider() {
        	
        	@Override
        	public void setSelection(ISelection selection) {
        		super.setSelection(selection);
        	}
        	
        });
        
    }

    @Override
    public void setFocus() {
    	if (root != null && root.getControl() != null && !root.getControl().isDisposed())
    		root.getControl().setFocus();
    	else if (!base.isDisposed())
    	    base.setFocus();
    }
    
}
