package org.simantics.scl.ui.imports.internal;

import gnu.trove.set.hash.THashSet;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;

import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.dialogs.TrayDialog;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.layout.TableColumnLayout;
import org.eclipse.jface.resource.FontDescriptor;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.EditingSupport;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.MenuAdapter;
import org.eclipse.swt.events.MenuEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.TableItem;
import org.osgi.framework.BundleContext;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;
import org.simantics.scl.compiler.commands.CommandSessionImportEntry;
import org.simantics.scl.ui.Activator;
import org.simantics.scl.ui.imports.ImportModulesAction;

public class ManageImportsDialog extends TrayDialog {

    private static final String DIALOG = "ManageImportsDialog"; //$NON-NLS-1$

    TableViewer tableViewer;
    ArrayList<CommandSessionImportEntry> imports;
    IDialogSettings dialogBoundsSettings;

    public ManageImportsDialog(Shell shell, ArrayList<CommandSessionImportEntry> imports) {
        super(shell);
        this.imports = imports;

        IDialogSettings settings = Activator.getInstance().getDialogSettings();
        dialogBoundsSettings = settings.getSection(DIALOG);
        if (dialogBoundsSettings == null)
            dialogBoundsSettings = settings.addNewSection(DIALOG);
    }

    @Override
    protected IDialogSettings getDialogBoundsSettings() {
        return dialogBoundsSettings;
    }

    @Override
    protected void configureShell(Shell newShell) {
        super.configureShell(newShell);
        newShell.setText("Manage Imports");
        newShell.setMinimumSize(800, 600);
    }

    @Override
    protected Control createDialogArea(Composite parent) {
        final Composite composite = (Composite) super.createDialogArea(parent);
        GridLayoutFactory.fillDefaults().margins(10,10).numColumns(2).applyTo(composite);
        GridDataFactory.fillDefaults().grab(true,true).applyTo(composite);

        Composite tableComposite = new Composite(composite, SWT.NONE);
        TableColumnLayout tcl = new TableColumnLayout();
        tableComposite.setLayout(tcl);
        GridDataFactory.fillDefaults().grab(true, true).applyTo(tableComposite);

        tableViewer = new TableViewer(tableComposite, SWT.BORDER | SWT.FULL_SELECTION | SWT.MULTI);

        // Colors
        Display display = getShell().getDisplay(); 
        final Color black = display.getSystemColor(SWT.COLOR_BLACK);
        final Color gray = display.getSystemColor(SWT.COLOR_GRAY);
        final Color red = display.getSystemColor(SWT.COLOR_RED);
        
        // Fonts
        final Font systemFont = display.getSystemFont();
        final Font bold = FontDescriptor.createFrom(systemFont).setStyle(SWT.BOLD).createFont(display);
        tableViewer.getTable().addDisposeListener(new DisposeListener() {
            @Override
            public void widgetDisposed(DisposeEvent e) {
                bold.dispose();
            }
        });
        
        // Column 1: module name
        TableViewerColumn column1 = new TableViewerColumn(tableViewer, SWT.NONE);
        column1.getColumn().setWidth(300);
        column1.getColumn().setText("Module");
        column1.setLabelProvider(new ColumnLabelProvider() {
            @Override
            public String getText(Object element) {
                return ((CommandSessionImportEntry)element).moduleName;
            }
            public Color getForeground(Object element) {
                CommandSessionImportEntry entry = (CommandSessionImportEntry)element;
                if(entry.disabled)
                    return gray;
                else if(entry.hasError)
                    return red;
                else
                    return black;
            }
            public Font getFont(Object element) {
                CommandSessionImportEntry entry = (CommandSessionImportEntry)element;
                if(entry.persistent)
                    return bold;
                else
                    return systemFont;
            }
        });
        column1.setEditingSupport(new EditingSupport(tableViewer) {
            CellEditor editor = new TextCellEditor(tableViewer.getTable());
            @Override
            protected CellEditor getCellEditor(Object element) {
                return editor;
            }
            @Override
            protected boolean canEdit(Object element) {
                return true;
            }
            @Override
            protected Object getValue(Object element) {
                return ((CommandSessionImportEntry)element).moduleName;
            }
            @Override
            protected void setValue(Object element, Object value) {
                ((CommandSessionImportEntry)element).moduleName = (String)value;
                getViewer().update(element, null);
            }
        });
        
        // Column 2: local name
        TableViewerColumn column2 = new TableViewerColumn(tableViewer, SWT.NONE);
        column2.getColumn().setWidth(100);
        column2.getColumn().setText("Local name");
        column2.setLabelProvider(new ColumnLabelProvider() {
            @Override
            public String getText(Object element) {
                return ((CommandSessionImportEntry)element).localName;
            }
        });
        column2.setEditingSupport(new EditingSupport(tableViewer) {
            CellEditor editor = new TextCellEditor(tableViewer.getTable());
            @Override
            protected CellEditor getCellEditor(Object element) {
                return editor;
            }
            @Override
            protected boolean canEdit(Object element) {
                return true;
            }
            @Override
            protected Object getValue(Object element) {
                return ((CommandSessionImportEntry)element).localName;
            }
            @Override
            protected void setValue(Object element, Object value) {
                ((CommandSessionImportEntry)element).localName = (String)value;
                getViewer().update(element, null);
            }
        });
        
        // Column 3: status
        TableViewerColumn column3 = new TableViewerColumn(tableViewer, SWT.NONE);
        column3.getColumn().setWidth(200);
        column3.getColumn().setText("Status");
        column3.setLabelProvider(new ColumnLabelProvider() {
            @Override
            public String getText(Object element) {
                return ((CommandSessionImportEntry)element).getStatusString();
            }
        });
        
        tcl.setColumnData(column1.getColumn(), new ColumnWeightData(5, 300));
        tcl.setColumnData(column2.getColumn(), new ColumnWeightData(1, 150));
        tcl.setColumnData(column3.getColumn(), new ColumnWeightData(1, 200));
        
        // Decorations
        tableViewer.getTable().setHeaderVisible(true);
        tableViewer.getTable().setLinesVisible(true);
        
        // Selection
        final Menu menu = new Menu(tableViewer.getTable());
        tableViewer.getTable().setMenu(menu);
        menu.addMenuListener(new MenuAdapter() {
            @Override
            public void menuShown(MenuEvent e) {
                MenuItem[] items = menu.getItems();
                for(int i=0;i < items.length;++i)
                    items[i].dispose();
                TableItem[] tableItems = tableViewer.getTable().getSelection();
                final CommandSessionImportEntry[] entries = new CommandSessionImportEntry[tableItems.length];
                for(int i=0;i<tableItems.length;++i)
                    entries[i] = (CommandSessionImportEntry)tableItems[i].getData();
                
                boolean hasPersistent = false;
                boolean hasTransient = false;
                boolean hasEnabled = false;
                boolean hasDisabled = false;
                
                for(CommandSessionImportEntry entry : entries) {
                    if(entry.persistent)
                        hasPersistent = true;
                    else
                        hasTransient = true;
                    if(entry.disabled)
                        hasDisabled = true;
                    else
                        hasEnabled = true;
                }
                
                {
                    MenuItem menuItem = new MenuItem(menu, SWT.NONE);
                    menuItem.setText("Delete");
                    menuItem.addSelectionListener(new SelectionAdapter() {
                        public void widgetSelected(SelectionEvent e) {
                            THashSet<CommandSessionImportEntry> removedEntries =
                                    new THashSet<CommandSessionImportEntry>(Arrays.asList(entries)); 
                            ArrayList<CommandSessionImportEntry> newImports = new ArrayList<CommandSessionImportEntry>(imports.size());
                            for(CommandSessionImportEntry entry : imports)
                                if(!removedEntries.contains(entry))
                                    newImports.add(entry);
                            imports = newImports;
                            setInput();
                        }
                    });
                }
                if(hasTransient) {
                    MenuItem menuItem = new MenuItem(menu, SWT.NONE);
                    menuItem.setText("Make persistent");
                    menuItem.addSelectionListener(new SelectionAdapter() {
                        public void widgetSelected(SelectionEvent e) {
                            for(CommandSessionImportEntry entry : entries)
                                entry.persistent = true;
                            setInput();
                        }
                    });
                }
                if(hasPersistent) {
                    MenuItem menuItem = new MenuItem(menu, SWT.NONE);
                    menuItem.setText("Make transient");
                    menuItem.addSelectionListener(new SelectionAdapter() {
                        public void widgetSelected(SelectionEvent e) {
                            for(CommandSessionImportEntry entry : entries)
                                entry.persistent = false;
                            setInput();
                        }
                    });
                }
                if(hasDisabled) {
                    MenuItem menuItem = new MenuItem(menu, SWT.NONE);
                    menuItem.setText("Enable");
                    menuItem.addSelectionListener(new SelectionAdapter() {
                        public void widgetSelected(SelectionEvent e) {
                            for(CommandSessionImportEntry entry : entries)
                                entry.disabled = false;
                            setInput();
                        }
                    });
                }
                if(hasEnabled) {
                    MenuItem menuItem = new MenuItem(menu, SWT.NONE);
                    menuItem.setText("Disable");
                    menuItem.addSelectionListener(new SelectionAdapter() {
                        public void widgetSelected(SelectionEvent e) {
                            for(CommandSessionImportEntry entry : entries)
                                entry.disabled = true;
                            setInput();
                        }
                    });
                }
            }
        });
        
        // Table content
        tableViewer.setContentProvider(ArrayContentProvider.getInstance());
        setInput();
        
        // Buttons
        Composite buttonComposite = new Composite(composite, SWT.NONE);
        GridDataFactory.fillDefaults().align(SWT.BEGINNING, SWT.BEGINNING).applyTo(buttonComposite);
        GridLayoutFactory.fillDefaults().applyTo(buttonComposite);
        
        BundleContext context = Activator.getInstance().getBundle().getBundleContext();
        ArrayList<ImportModulesAction> actions = new ArrayList<ImportModulesAction>();
        try {
            for(ServiceReference<ImportModulesAction> ref : 
                context.getServiceReferences(ImportModulesAction.class, null)) {
                ImportModulesAction action = context.getService(ref);
                actions.add(action);
            }
        } catch (InvalidSyntaxException e1) {
            e1.printStackTrace();
        }
        
        Collections.sort(actions);
        
        for(final ImportModulesAction action : actions) {
            Button button = new Button(buttonComposite, SWT.NONE);
            GridDataFactory.fillDefaults().applyTo(button);
            button.setText(action.name);
            button.addSelectionListener(new SelectionAdapter() {
                @Override
                public void widgetSelected(SelectionEvent e) {
                    if(action.editImports(getShell(), imports))
                        setInput();
                }
            });
        }
        return composite;
    }
    
    private void setInput() {
        CommandSessionImportEntry[] array = imports.toArray(new CommandSessionImportEntry[imports.size()]);
        Arrays.sort(array);
        tableViewer.setInput(array);
    }

    public ArrayList<CommandSessionImportEntry> getImports() {
        return imports;
    }
    
    @Override
    protected boolean isResizable() {
        return true;
    }
}
