/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.tests;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;

import junit.framework.TestCase;

import org.simantics.databoard.util.binary.BinaryFile;
import org.simantics.databoard.util.binary.BinaryReadable;
import org.simantics.databoard.util.binary.BinaryWriteable;
import org.simantics.databoard.util.binary.InputStreamReadable;
import org.simantics.databoard.util.binary.OutputStreamWriteable;

public class TestWriteable extends TestCase {

	public static final int ITERATIONS = 50;
	
	public void testWriteRead() 
	throws FileNotFoundException, IOException 
	{		
		File tmp1 = File.createTempFile("xyz1", "tmp");
		File tmp2 = File.createTempFile("xyz2", "tmp");
		File tmp3 = File.createTempFile("xyz3", "tmp");
		File tmp4 = File.createTempFile("xyz4", "tmp");
		
		tmp1.deleteOnExit();
		tmp2.deleteOnExit();
		tmp3.deleteOnExit();
		tmp4.deleteOnExit();
		
		BinaryFile      write1 = new BinaryFile( new RandomAccessFile(tmp1, "rw") );
		BinaryWriteable write2 = new OutputStreamWriteable( new FileOutputStream( tmp2, true  ) ); 		
		BinaryWriteable write3 = new OutputStreamWriteable( new BufferedOutputStream( new FileOutputStream( tmp3, true  ) ) );
		BinaryFile write4 = new BinaryFile( tmp4 );
		
		System.out.println("Write Times:");
		long time2 = measureWrite(write2);
		long time3 = measureWrite(write3);
		long time1 = measureWrite(write1);
		long time4 = measureWrite(write4);
		
		System.out.println();
		System.out.println("RandomAccessFile: "+time1);
		System.out.println("FileOutputStream: "+time2);
		System.out.println("FileOutputStream (Buffered) : "+time3);
		System.out.println("BinaryFile : "+time4);

		write4.close();
		
		BinaryFile     read1 = new BinaryFile( new RandomAccessFile(tmp1, "r") );
		BinaryReadable read2 = new InputStreamReadable( new FileInputStream( tmp2 ), tmp1.length() ); 		
		BinaryReadable read3 = new InputStreamReadable( new BufferedInputStream( new FileInputStream( tmp3 ) ), tmp1.length() );
		BinaryFile     read4 = new BinaryFile( tmp4 );
		
		System.out.println("Read Times:");
		time2 = measureRead(read2);
		time3 = measureRead(read3);				
		time1 = measureRead(read1);
		time4 = measureRead(read4);				
		System.out.println();
		System.out.println("RandomAccessFile: "+time1);
		System.out.println("FileOutputStream: "+time2);
		System.out.println("FileOutputStream (Buffered) : "+time3);
		System.out.println("BinaryFile : "+time4);
		
		// re-write
		write1.position(0);
		time1 = measureWrite(write1);		
		// Re-read
		read1.position(0);
		time2 = measureRead(read1);
		System.out.println("\nRandomAccessFile (again): write="+time1+", read="+time2);
		
	}
	
	static long measureWrite(BinaryWriteable write) 
	throws IOException
	{		
		System.gc();
		byte[] data = new byte[1024];
		for (int i=0; i<data.length; i++)
			data[i] = (byte) i;
		
		long startTime = System.currentTimeMillis();		
		for (int iter=0; iter<ITERATIONS; iter++)
		{
			if (iter % 10==0) System.out.print(".");
			for (int i=0; i<256; i++)
				write.write((byte)i);
			for (int i=0; i<256; i++)
				write.writeDouble(i);
			for (int i=0; i<256; i++)
				write.writeFloat(i);
			for (int i=0; i<256; i++)
				write.writeInt(i);
			for (int i=0; i<256; i++)
				write.writeShort((short)i);
			write.write(data);			
		}		
		write.flush();
		long elapsedTime = System.currentTimeMillis() - startTime;		
		
		return elapsedTime;
	}
	
	static long measureRead(BinaryReadable read) 
	throws IOException
	{		
		System.gc();
		byte[] data = new byte[1024];
		
		long startTime = System.currentTimeMillis();		
		for (int iter=0; iter<ITERATIONS; iter++)
		{
			if (iter % 10==0) System.out.print(".");
			for (int i=0; i<256; i++) 
				assertEquals((byte) i, read.readByte());
			for (int i=0; i<256; i++)
				assertEquals((double) i, read.readDouble());
			for (int i=0; i<256; i++)
				assertEquals((float) i, read.readFloat());
			for (int i=0; i<256; i++)
				assertEquals((int) i, read.readInt());
			for (int i=0; i<256; i++)
				assertEquals((short) i, read.readShort());
			read.readFully(data);
			for (int i=0; i<data.length; i++)
				assertEquals(data[i], (byte) i);
		}		
		long elapsedTime = System.currentTimeMillis() - startTime;
		
		return elapsedTime;
	}
	
	
}

