package org.simantics.databoard.serialization.impl;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.util.List;

import org.simantics.databoard.binding.ArrayBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.serialization.SerializationException;
import org.simantics.databoard.serialization.Serializer.NonRecursiveSerializer;
import org.simantics.databoard.type.ArrayType;
import org.simantics.databoard.util.Range;

public class LongArraySerializer extends NonRecursiveSerializer {

	Range length;
	Integer fixedLength, fixedSize;
	
	public LongArraySerializer(ArrayBinding binding)
	{
		ArrayType arrayType = (ArrayType) binding.type();
		this.length = arrayType.getLength();
		if (length!=null && length.getLower().equals(length.getUpper()) && length.getLower().getValue()!=null)
		{
			fixedLength = length.getLower().getValue().intValue();
			fixedSize = fixedLength * 8;
		}
	}
	
	@Override
	public Object deserialize(DataInput in)
	throws IOException 
	{
		int length = fixedSize != null ? fixedLength : in.readInt();			
		if (length<0) throw new SerializationException("Cannot use negative array length");
		assertRemainingBytes(in, length*8L);					

		long[] array = new long[length];
		for(int i=0;i<array.length;++i)
			array[i] = in.readLong();
		return array;
	}
	
	public Object deserializeToTry(DataInput in, List<Object> identities, Object obj) throws IOException
	{
		int length = fixedLength != null ? fixedLength : in.readInt();
		long[] array = (long[]) obj;
		if (length!=array.length) array = new long[ length ];
		assertRemainingBytes(in, length*8L);					
		
		for (int i=0; i<array.length;i++)
			array[i] = in.readLong();
		
		return array;
	}
	

	@Override
	public void deserializeTo(DataInput in, Object obj) throws IOException {
		try {
			int length = fixedLength != null ? fixedLength : in.readInt();
			long[] array = (long[]) obj;
			if (length!=array.length) throw new BindingException("primitive array is size immutable");
			assertRemainingBytes(in, length*8L);					
			for (int i=0; i<array.length;i++)
				array[i] = in.readLong();
		} catch (BindingException e) {
			throw new IOException( e ); 
		}
	}

	@Override
	public void skip(DataInput in)
	throws IOException {
		int length = fixedSize != null ? fixedLength : in.readInt();			
		in.skipBytes(8* length);			
	}
	
	@Override
	public void serialize(DataOutput out, Object obj)
	throws IOException 
	{
		long[] array = (long[])obj;
		if (fixedSize==null) 
			out.writeInt(array.length);
		for(long f : array)
			out.writeLong(f);
	}

	@Override
	public Integer getConstantSize() {
		return fixedSize;
	}

	@Override
	public int getSize(Object obj) {
		if (fixedSize!=null) return fixedSize;
		long[] array = (long[])obj;			
		return 4 + 8 * array.length;
	}
	
	@Override
	public int getMinSize() {
		return fixedSize != null ? fixedSize : 4;
	}
	
}
