/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding.impl;

import org.simantics.databoard.binding.FloatBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.error.UnsupportedOperationException;
import org.simantics.databoard.type.FloatType;

/**
 * Binds FloatType to java.lang.Float-class.
 *
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public class FloatBindingDefault extends FloatBinding {

	public FloatBindingDefault(FloatType type) {
		super(type);
	}

	public Object create(Float value) {
		return value;
	}
	
	public Object create(float value) {
		return Float.valueOf(value);
	}
	
	@Override
	public Object create(Number value) 
	{
		if (value.getClass()==Float.class) return value;
		return Float.valueOf( value.floatValue() );
	}
    
	@Override
	public Object create(String value) throws BindingException {
		try  {
			return Float.parseFloat(value);
		} catch (java.lang.NumberFormatException e) {
			throw new BindingException( e );
		}
	}
	
    @Override
	public boolean isInstance(Object obj) {
		return obj instanceof Float;
	}
	
	public Float getValue(Object o) 
	throws BindingException {
		if (o.getClass()!=Float.class)
			throw new BindingException("Unexpected class "+o.getClass().getSimpleName()+", Float expected");		
		return ((Float)o);
	}

	public float getValue_(Object o) throws BindingException {
		if (o.getClass()!=Float.class)
			throw new BindingException("Unexpected class "+o.getClass().getSimpleName()+", Float expected");		
		return ((Float)o);		
	}
	
	@Override
	public void setValue(Object obj, Number value) throws BindingException {
		throw new UnsupportedOperationException("Cannot change the value of immutable java.lang.Float");
	}

	public void setValue(Object obj, float value) throws BindingException {
		throw new UnsupportedOperationException("Cannot change the value of immutable java.lang.Float");
	}
	
	@Override
	public boolean isImmutable() {
		return true;
	}
	
    @Override
    public int compare(Object o1, Object o2)
    {
    	Float d1 = (Float) o1;
    	Float d2 = (Float) o2;
    	return d1.compareTo(d2);
    }
	
	
}

