/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.utils;

import java.util.Collection;

/**
 * GroupException is a Exception container for a set of exceptions.
 * 
 * <p>
 * Use {@link #group(Collection)} or {@link #group(Exception...)} to construct
 * it.
 * 
 * @author Toni Kalajainen
 */
public class GroupException extends Exception {

    private static final long serialVersionUID = 1L;

    private final Throwable errors[];

    private GroupException(Throwable[] errors) {
        super(GroupException.getAsText(errors));
        this.errors = errors;
    }

    public static Exception group(Exception... errors) {
        if (errors.length == 0)
            throw new IllegalArgumentException("zero exceptions for GroupException");
        if (errors.length == 1)
            return errors[0];
        return new GroupException(errors);
    }

    public static Throwable group(Collection<? extends Throwable> collection) {
        if (collection.size() == 0)
            throw new IllegalArgumentException("zero exceptions for GroupException");
        if (collection.size() == 1)
            return collection.iterator().next();
        return new GroupException(collection.toArray(new Exception[collection.size()]));
    }

    private static String getAsText(Throwable errors[]) {
        String result = "";
        for (Throwable error : errors) {
            result += error.getClass().getName()+": "+error.getMessage() + "\n";
        }
        return result;
    }

    public Throwable[] getErrors() {
        return errors;
    }

}
