/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.synchronization.graph;

import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;

import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.CommentMetadata;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.synchronization.ModificationAdapter;
import org.simantics.g2d.element.ElementHints;
import org.simantics.g2d.element.IElement;

/**
 * @author Tuukka Lehtonen
 */
public class TranslateElement extends ModificationAdapter {

    Resource element;
    Point2D  absolutePosition;
    Point2D  offset;

    public static TranslateElement absolute(Resource element, double x, double y) {
        return new TranslateElement(element, new Point2D.Double(x, y), null);
    }

    public static TranslateElement absolute(Resource element, Point2D absolutePosition) {
        return new TranslateElement(element, absolutePosition, null);
    }

    public static TranslateElement offset(Resource element, double x, double y) {
        return new TranslateElement(element, null, new Point2D.Double(x, y));
    }

    public static TranslateElement offset(Resource element, Point2D offset) {
        return new TranslateElement(element, null, offset);
    }

    /**
     * @param element the element to translate, must contain a {@link Resource}
     *        type {@link ElementHints#KEY_OBJECT} hint
     * @param absolutePosition <code>null</code> to not define an absolute
     *        position to set, otherwise this position will override the
     *        original translation of the element
     * @param offset an offset to apply to the element's position. If
     *        {@link #absolutePosition} is null, only an offset will be applied
     */
    public TranslateElement(IElement element, Point2D absolutePosition, Point2D offset) {
        this((Resource) element.getHint(ElementHints.KEY_OBJECT), absolutePosition, offset);
    }

    /**
     * @param element the element to translate
     * @param absolutePosition <code>null</code> to not define an absolute
     *        position to set, otherwise this position will override the
     *        original translation of the element
     * @param offset an offset to apply to the element's position. If
     *        {@link #absolutePosition} is null, only an offset will be applied
     */
    public TranslateElement(Resource element, Point2D absolutePosition, Point2D offset) {
        super(LOW_PRIORITY);
        this.element = element;
        this.absolutePosition = absolutePosition;
        this.offset = offset;
    }

    @Override
    public void perform(WriteGraph graph) throws DatabaseException {
        if (absolutePosition == null && offset == null)
            return;

        AffineTransform at = DiagramGraphUtil.getTransform(graph, element);

        double x = at.getTranslateX();
        double y = at.getTranslateY();
        if (absolutePosition != null) {
            x = absolutePosition.getX();
            y = absolutePosition.getY();
        }
        if (offset != null) {
            x += offset.getX();
            y += offset.getY();
        }

        boolean changed = !(x == at.getTranslateX() && y == at.getTranslateY());
        if (!changed)
            return;

        at.setTransform(at.getScaleX(), at.getShearX(), at.getShearY(), at.getScaleY(), x, y);

        DiagramGraphUtil.setTransform(graph, element, at);
        CommentMetadata cm = graph.getMetadata(CommentMetadata.class);
        graph.addMetadata(cm.add("Translated " + element));
    }

}
