/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.symbolcontribution;

import org.simantics.db.ReadGraph;
import org.simantics.db.RequestProcessor;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ResourceRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.Listener;
import org.simantics.db.request.Read;
import org.simantics.diagram.query.DiagramRequests;
import org.simantics.diagram.symbollibrary.ISymbolGroup;
import org.simantics.diagram.symbollibrary.ISymbolItem;
import org.simantics.diagram.synchronization.graph.BasicResources;
import org.simantics.g2d.element.ElementClass;
import org.simantics.g2d.element.handler.StaticSymbol;
import org.simantics.ui.SimanticsUI;
import org.simantics.utils.datastructures.cache.ProvisionException;
import org.simantics.utils.datastructures.hints.IHintObservable;

/**
 * @author Tuukka Lehtonen
 */
public class ElementSymbolItem extends SymbolItem {

    public ElementSymbolItem(Resource item, String name, ISymbolGroup group) {
        super(
                AdaptableTriple.make(
                        item,
                        // Prevents infinite recursion which will happen
                        // if SymbolGroup is used as group ID.
                        group instanceof IIdentifiedObject ? ((IIdentifiedObject) group).getId() : group,
                        // Name is in the identification so that name
                        // changes refresh the symbol library
                        name),
                name,
                group);
    }

    public ElementSymbolItem(Resource item, String name, String description, ISymbolGroup group) {
        super(
                AdaptableTriple.make(
                        item,
                        // Prevents infinite recursion which will happen
                        // if SymbolGroup is used as group ID.
                        group instanceof IIdentifiedObject ? ((IIdentifiedObject) group).getId() : group,
                        // Name is in the identification so that name
                        // changes refresh the symbol library
                        name),
                name,
                description,
                group);
    }

    @Override
    public ElementClass getElementClass(IHintObservable hints) {
        Resource item = adapt(Resource.class);
        RequestProcessor sgrp = SimanticsUI.peekSession();
        if (sgrp == null)
            throw new ProvisionException("No RequestProcessor available for querying an ElementClass for resource " + item);
        try {
            Listener<ElementClass> listener = hints.getHint(ISymbolItem.KEY_ELEMENT_CLASS_LISTENER);
            ElementClass ec = listener == null
                    ? sgrp.syncRequest(createRequest(item, hints))
                    : sgrp.syncRequest(createRequest(item, hints), listener);
            if (ec == null)
                throw new ProvisionException("ElementClass query failed, returned null");
            if (!ec.containsClass(StaticSymbol.class))
                throw new ProvisionException("ElementClass " + ec + " does not provide a StaticSymbol handler");
            return ec;
        } catch (DatabaseException e) {
            throw new ProvisionException(e);
        }
    }

    private Read<ElementClass> createRequest(Resource item, final IHintObservable hints) {
        return new ResourceRead<ElementClass>(item) {
            @Override
            public ElementClass perform(ReadGraph graph) throws DatabaseException {
                BasicResources.getInstance(graph);
                return graph.syncRequest( DiagramRequests.getElementClass(resource, hints) );
            }
        };
    }

}
