/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.query;

import java.util.Set;

import org.eclipse.core.runtime.IProgressMonitor;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Statement;
import org.simantics.db.common.ResourceArray;
import org.simantics.db.common.request.ResourceRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.Listener;
import org.simantics.db.request.AsyncRead;
import org.simantics.db.request.Read;
import org.simantics.diagram.adapter.IDiagramLoader;
import org.simantics.diagram.adapter.NodeRequest;
import org.simantics.diagram.connection.ConnectionVisuals;
import org.simantics.diagram.content.TerminalMap;
import org.simantics.diagram.flag.FlagUtil;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.diagram.synchronization.graph.DiagramGraphUtil;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.element.ElementClass;
import org.simantics.g2d.element.IElement;
import org.simantics.g2d.page.DiagramDesc;
import org.simantics.structural.stubs.StructuralResource2;
import org.simantics.structural2.modelingRules.CPConnection;
import org.simantics.structural2.modelingRules.CPTerminal;
import org.simantics.structural2.modelingRules.IConnectionPoint;
import org.simantics.structural2.modelingRules.IModelingRules;
import org.simantics.utils.datastructures.hints.IHintObservable;
import org.simantics.utils.page.PageDesc;

/**
 * @author Tuukka Lehtonen
 */
public class DiagramRequests {

    /**
     * @param diagram
     * @return
     */
    public static Read<PageDesc> getPageDesc(final Resource diagram, final PageDesc defaultValue) {
        return new Read<PageDesc>() {
            @Override
            public PageDesc perform(ReadGraph graph) throws DatabaseException {
                return DiagramGraphUtil.getPageDesc(graph, diagram, defaultValue);
            }
        };
    }

    public static Read<DiagramDesc> getDiagramDesc(final Resource diagram) {
        return new DiagramDescRead(diagram);
    }

    /**
     * Adapt a diagram Element type resource into an {@link ElementClass}.
     * 
     * @param elementType
     * @param hints
     * @return
     */
    public static AsyncRead<ElementClass> getElementClass(Resource elementType, IHintObservable hints) {
        return new ElementClassQuery(elementType, hints);
    }

    /**
     * @param canvas
     * @param diagram
     * @param element
     * @param loadListener
     * @return
     */
    public static AsyncRead<IElement> getElement(ICanvasContext canvas, IDiagram diagram, Resource element, Listener<IElement> loadListener) {
        return new NodeRequest(canvas, diagram, element, loadListener);
    }

    /**
     * @param monitor
     * @param resource
     * @param structuralPath
     * @param loader
     * @param initialHints
     * @return
     */
    public static Read<IDiagram> loadDiagram(IProgressMonitor monitor, Resource model, Resource resource, Resource runtime, ResourceArray structuralPath,
            IDiagramLoader loader, IHintObservable initialHints) {
        return new DiagramLoadQuery(monitor, model, resource, runtime, structuralPath, loader, initialHints);
    }

    /**
     * @param diagram
     * @return
     */
    public static Read<IModelingRules> getModelingRules(final Resource diagram, final IModelingRules defaultValue) {
        return new Read<IModelingRules>() {
            @Override
            public IModelingRules perform(ReadGraph graph) throws DatabaseException {
                StructuralResource2 sr = StructuralResource2.getInstance(graph);
                Resource rules = graph.getPossibleObject(diagram, sr.HasModelingRules);
                if (rules == null)
                    return defaultValue;
                return graph.adapt(rules, IModelingRules.class);
            }
        };
    }

    /**
     * @param connectionType
     * @return
     */
    public static Read<ConnectionVisuals> getConnectionVisuals(Resource connectionType) {
        return new ConnectionVisualsRequest(connectionType);
    }

    /**
     * @param flag
     * @return
     */
    public static Read<String[]> getFlagText(Resource flag) {
        return new FlagTextQuery(flag);
    }

    public static Read<Resource> getFlagPair(Resource flag) {
        return new FlagPairQuery(flag);
    }

    /**
     * @param element
     * @return
     */
    public static Read<TerminalMap> elementTypeTerminals(Resource elementType) {
        return new ResourceRead<TerminalMap>(elementType) {
            @Override
            public TerminalMap perform(ReadGraph graph) throws DatabaseException {
                return DiagramGraphUtil.getElementTypeTerminals(graph, resource);
            }
        };
    }

    /**
     * @param element
     * @return
     */
    public static Read<TerminalMap> elementTerminals(Resource element) {
        return new ResourceRead<TerminalMap>(element) {
            @Override
            public TerminalMap perform(ReadGraph graph) throws DatabaseException {
                return DiagramGraphUtil.getElementTerminals(graph, resource);
            }
        };
    }

    /**
     * Browses all diagram connections that are related to {@code connection} by flags.
     * Adds the connections to set {@code conns} and all related connection points to set {@code cps}.
     */
	public static void expandConnections(ReadGraph graph, Resource connection, Set<Resource> conns, Set<IConnectionPoint> cps) throws DatabaseException {

		if(!conns.add(connection)) return;

		DiagramResource DIA = DiagramResource.getInstance(graph);
		StructuralResource2 STR = StructuralResource2.getInstance(graph);

		cps.add(new CPConnection(connection));
		for(Resource connector : graph.getObjects(connection, DIA.HasConnector)) {
			for(Statement stm : graph.getStatements(connector, STR.Connects)) {
				Resource element = stm.getObject();
				if(element.equals(connection)) continue;
				if(graph.isInstanceOf(element, DIA.Flag)) {
					FlagUtil.forCounterparts(graph, element, (otherFlag) -> {
						Resource flagConnector = graph.getPossibleObject(otherFlag, DIA.Flag_ConnectionPoint);
						if(flagConnector != null) {
							Resource otherConnection = graph.getPossibleObject(flagConnector, DIA.IsConnectorOf);
							if(otherConnection != null) {
								expandConnections(graph, otherConnection, conns, cps);
							}
						}
					});
				} else {
					Resource connectionPoint = graph.getInverse(stm.getPredicate());
					//System.err.println("term: " + NameUtils.getSafeName(graph, element) + " : " + NameUtils.getSafeName(graph, connectionPoint));
					cps.add(new CPTerminal(element, connectionPoint));
				}
			}
		}

	}
    
}
