/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.participant;

import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.Collection;

import org.simantics.Simantics;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.CommentMetadata;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.content.ConnectionUtil;
import org.simantics.diagram.elements.ElementTransforms;
import org.simantics.diagram.elements.ElementTransforms.TransformedObject;
import org.simantics.diagram.flag.IOTableUtil;
import org.simantics.diagram.flag.IOTablesInfo;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.diagram.ui.DiagramModelHints;
import org.simantics.g2d.canvas.Hints;
import org.simantics.g2d.diagram.participant.pointertool.TranslateMode;
import org.simantics.g2d.element.ElementHints;
import org.simantics.g2d.element.ElementUtils;
import org.simantics.g2d.element.IElement;
import org.simantics.g2d.element.handler.Move;
import org.simantics.scenegraph.g2d.IG2DNode;
import org.simantics.scenegraph.g2d.nodes.Decoration;
import org.simantics.scenegraph.utils.NodeUtil;
import org.simantics.utils.ui.ErrorLogger;

/**
 * This participant handles the diagram in translate elements mode with support
 * for route graph connections.
 * 
 * @author Tuukka Lehtonen
 */
public class TranslateMode2 extends TranslateMode {

    public TranslateMode2(Point2D startingPoint, Point2D currentPoint, int mouseId, Collection<IElement> elements) {
        super(startingPoint, currentPoint, mouseId, elements);
    }

    @Override
    protected boolean commit() {
        for (IElement el : elementsToReallyTranslate) {
            Move move = el.getElementClass().getAtMostOneItemOfClass(Move.class);
            if (move != null) {
                Point2D oldPos = move.getPosition(el);
                move.moveTo(el, oldPos.getX() + dx, oldPos.getY() + dy);
            }
        }
 
        try {
            Simantics.getSession().syncRequest(new WriteRequest() {
                @Override
                public void perform(WriteGraph graph) throws DatabaseException {
                    DiagramResource DIA = DiagramResource.getInstance(graph);

                    Collection<TransformedObject> transformed = new ArrayList<TransformedObject>();
                    for (IElement e : elementsToReallyTranslate) {
                        Object obj = ElementUtils.getObject(e);
                        if (obj instanceof Resource) {
                            Resource r = (Resource) obj;
                            if (graph.isInstanceOf(r, DIA.RouteGraphConnection))
                                continue;
                            if (graph.isInstanceOf(r, DIA.Flag)) {
                                if(handleFlag(graph, DIA, e, r))
                                    continue;
                            }
                            AffineTransform at = ElementUtils.getLocalTransform(e, new AffineTransform());
                            transformed.add( new TransformedObject((Resource) obj, at) );
                        }
                    }

                    if (!transformed.isEmpty()) {
                        CommentMetadata cm = graph.getMetadata(CommentMetadata.class);
                        graph.addMetadata(cm.add("Translated " + transformed.size() + " "
                                + (transformed.size() == 1 ? "element" : "elements")
                                + " by (" + dx + "," + dy + ") mm."));
                        graph.markUndoPoint();
                    }

                    // Normal transforms
                    ElementTransforms.setTransformRequest(transformed).perform(graph);

                    // Custom handling for moving interior route nodes
                    ConnectionUtil cu = new ConnectionUtil(graph);
                    for (IElement c : translatedConnections) {
                        Object obj = (Object) ElementUtils.getObject(c);
                        if (obj instanceof Resource) {
                            Resource connection = (Resource) obj;
                            if (graph.isInstanceOf(connection, DIA.RouteGraphConnection)) {
                                cu.translateRouteNodes(connection, dx, dy);
                            }
                        }
                    }
                }
            });
        } catch (DatabaseException err) {
            ErrorLogger.defaultLogError(err);
        }

        for (IElement dirty : elementsToDirty)
            dirty.setHint(Hints.KEY_DIRTY, Hints.VALUE_SG_DIRTY);

        setDirty();
        remove();
        return false;
    }
    
    IOTablesInfo ioTablesInfo = null;
    
    private boolean handleFlag(WriteGraph graph, DiagramResource DIA,
            IElement flagElement, Resource flagResource) throws DatabaseException {
        if(ioTablesInfo == null)
            ioTablesInfo = IOTableUtil.getIOTablesInfo(graph, 
                    (Resource)diagram.getHint(DiagramModelHints.KEY_DIAGRAM_RESOURCE));
        
        // Use the center of the flag for determining the table binding
        Object sgNode = flagElement.getHint(ElementHints.KEY_SG_NODE);
        if(!(sgNode instanceof IG2DNode))
            return false;
        Rectangle2D flagBounds = NodeUtil.getLocalBounds((IG2DNode)sgNode, Decoration.class);
        Point2D translateVector = getTranslateVector();
        double flagX = flagBounds.getCenterX()+translateVector.getX();
        double flagY = flagBounds.getCenterY()+translateVector.getY();
        return ioTablesInfo.updateBinding(graph, DIA, flagResource, flagX, flagY);
    }

}
