/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.internal.timing;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.internal.DebugPolicy;
import org.simantics.diagram.synchronization.ErrorHandler;

/**
 * @author Tuukka Lehtonen
 */
public final class Timing {

    private static Method BEGIN_METHOD;
    private static Method END_METHOD;

    static {
        // Try to get ThreadLog, if possible.
        try {
            Class<?> logClass = Class.forName("org.simantics.threadlog.ThreadLog");
            Class<?> taskClass = Class.forName("org.simantics.threadlog.Task");
            Method begin = logClass.getMethod("BEGIN", String.class);
            Method end = taskClass.getMethod("end");

            BEGIN_METHOD = begin;
            END_METHOD = end;
        } catch (ClassNotFoundException e) {
        } catch (NoSuchMethodException e) {
        } catch (SecurityException e) {
            // No dice, just don't do timing then.
        }
    }

    public static void timed(String label, GTask r) throws DatabaseException {
        Object t = Timing.BEGIN(label);
        try {
            long start = 0, end = 0;
            if (DebugPolicy.PERFORM_TIMING)
                start = System.nanoTime();
            r.run();
            if (DebugPolicy.PERFORM_TIMING) {
                end = System.nanoTime();
                System.out.println("TIMED " + ((end-start)*1e-6) + "ms for " + label);
            }
        } finally {
            Timing.END(t);
        }
    }

    public static void safeTimed(ErrorHandler errorHandler, String label, GTask r) {
        Object t = Timing.BEGIN(label);
        try {
            long start = 0, end = 0;
            if (DebugPolicy.PERFORM_TIMING)
                start = System.nanoTime();
            try {
                r.run();
            } catch (DatabaseException e) {
                errorHandler.error(e.getMessage(), e);
            }
            if (DebugPolicy.PERFORM_TIMING) {
                end = System.nanoTime();
                System.out.println("TIMED " + ((end-start)*1e-6) + "ms for " + label);
            }
        } finally {
            Timing.END(t);
        }
    }

    public static Object BEGIN(String name) {
        if (DebugPolicy.PERFORM_TIMING) {
            //return ThreadLog.BEGIN(name);
            if (BEGIN_METHOD != null) {
                try {
                    return BEGIN_METHOD.invoke(null, name);
                } catch (IllegalArgumentException e) {
                    e.printStackTrace();
                } catch (IllegalAccessException e) {
                    e.printStackTrace();
                } catch (InvocationTargetException e) {
                    e.printStackTrace();
                }
            }
        }
        return null;
    }

    public static void END(Object task) {
        if (DebugPolicy.PERFORM_TIMING) {
            //((Task) task).end();
            if (task != null) {
                try {
                    END_METHOD.invoke(task);
                } catch (IllegalArgumentException e) {
                    e.printStackTrace();
                } catch (IllegalAccessException e) {
                    e.printStackTrace();
                } catch (InvocationTargetException e) {
                    e.printStackTrace();
                }
            }
        }
    }

}
