/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.flag;

import org.eclipse.core.expressions.PropertyTester;
import org.simantics.DatabaseJob;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.common.utils.RequestUtil;
import org.simantics.db.exception.DatabaseException;
import org.simantics.ui.SimanticsUI;
import org.simantics.ui.utils.ResourceAdaptionUtils;

/**
 * An Eclipse property tester for flag elements.
 * 
 * @author Tuukka Lehtonen
 */
public class FlagPropertyTester extends PropertyTester {

    /**
     * Tests if the received flag resource is connected to a counterpart.
     * Default expected value is true.
     */
    private static final String CONNECTED = "connected";

    /**
     * Tests if the received flag resource is connected to a counterpart.
     * Default expected value is true.
     */
    private static final String LOCALLY_CONNECTED = "locallyConnected";

    /**
     * Tests if the received flag resource is marked as an external flag.
     */
    private static final String EXTERNAL = "external";

    /**
     * Tests if the received flag resource is a merged flag with more than 1
     * counterpart.
     */
    private static final String MERGED = "merged";

    @Override
    public boolean test(Object receiver, final String property, final Object[] args, final Object expectedValue) {
        final Resource resource = ResourceAdaptionUtils.toSingleResource(receiver);
        if (resource == null)
            return false;

        Session session = SimanticsUI.peekSession();
        if (session == null)
            return false;

        if (DatabaseJob.inProgress())
            return false;

        try {
            return RequestUtil.trySyncRequest(
                    session,
                    SimanticsUI.UI_THREAD_REQUEST_START_TIMEOUT,
                    SimanticsUI.UI_THREAD_REQUEST_EXECUTION_TIMEOUT,
                    false,
                    new UniqueRead<Boolean>() {
                @Override
                public Boolean perform(ReadGraph g) throws DatabaseException {
                    return Boolean.valueOf(doTest(g, resource, property, args, expectedValue));
                }
            });
        } catch (DatabaseException | InterruptedException e) {
            // Purposefully not logging these exceptions, there might be way too
            // many even under normal circumstances.
            // TODO: add debug tracing options controlling the printing of these exceptions
            return false;
        }
    }

    private boolean doTest(ReadGraph graph, Resource resource, String property, Object[] args, Object expectedValue) throws DatabaseException {
        if (EXTERNAL.equals(property)) {
            boolean expected = parseBoolean(expectedValue, true);
            boolean ext = FlagUtil.isExternal(graph, resource);
            return ext ? expected : expected;
        } else if (CONNECTED.equals(property)) {
            boolean expected = parseBoolean(expectedValue, true);
            boolean isConnected = 
                FlagUtil.countCounterparts(graph, resource) > 0 ||
                FlagUtil.isLifted(graph, resource);
            return isConnected == expected;
        } else if (LOCALLY_CONNECTED.equals(property)) {
            boolean expected = parseBoolean(expectedValue, true);
            boolean isLocallyConnected = 
                FlagUtil.isJoinedInSingleDiagram(graph, resource);
            return isLocallyConnected == expected;
        } else if (MERGED.equals(property)) {
            boolean expected = parseBoolean(expectedValue, true);
            boolean isMerged = FlagUtil.countCounterparts(graph, resource) > 1;
            return isMerged == expected;
        }
        return false;
    }

    boolean parseBoolean(Object value, boolean defaultValue) {
        if (value instanceof Boolean)
            return (Boolean) value;
        if (value instanceof String)
            return Boolean.parseBoolean((String) value);
        return defaultValue;
    }

}
