/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.adapter;

import gnu.trove.list.array.TIntArrayList;
import gnu.trove.map.hash.THashMap;
import gnu.trove.procedure.TIntProcedure;
import gnu.trove.set.hash.THashSet;

import java.util.ArrayList;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;

import org.simantics.db.AsyncReadGraph;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.procedure.adapter.ProcedureAdapter;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.procedure.AsyncMultiProcedure;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.diagram.content.ConnectionPartData;
import org.simantics.diagram.content.ConnectionPartRequest;
import org.simantics.diagram.content.DiagramContents;
import org.simantics.diagram.content.EdgeResource;
import org.simantics.diagram.content.RouteGraphConnectionPartData;
import org.simantics.diagram.content.RouteGraphConnectionPartRequest;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.diagram.synchronization.ErrorHandler;
import org.simantics.g2d.canvas.ICanvasContext;

/**
 * @author Tuukka Lehtonen
 */
public class DiagramContentRequest extends BaseRequest<Resource, DiagramContents> {

    int previousElementCount = 32;
    ErrorHandler errorHandler;

    public DiagramContentRequest(ICanvasContext canvas, Resource resource, ErrorHandler errorHandler) {
        super(canvas, resource);
        this.errorHandler = errorHandler;
    }

    @Override
    public DiagramContents perform(ReadGraph g) throws DatabaseException {

        final DiagramResource DIA = DiagramResource.getInstance(g);
        final DiagramContents result = new DiagramContents();

        result.elements =   new ArrayList<Resource>(previousElementCount);
        result.nodeSet = new THashSet<Resource>();
        result.connectionSet = new THashSet<Resource>();
        result.connectionSegments = new THashSet<EdgeResource>();
        result.branchPoints = new THashSet<Resource>();
        result.routeGraphConnectionSet = new THashSet<Resource>();
        result.routeLinks = new THashSet<EdgeResource>();
        result.routeLines = new THashSet<Resource>();
        result.routePoints = new THashSet<Resource>();

        result.partToConnection = new THashMap<Object, Resource>();

        // These help loading result.elements in the correct order.
        final AtomicInteger index = new AtomicInteger();
        final TIntArrayList unrecognizedElementIndices = new TIntArrayList();

        g.forOrderedSet(data, new AsyncMultiProcedure<Resource>() {

            @Override
            public void execute(AsyncReadGraph graph, final Resource component) {

                // Must add the elements to the result set here in order to
                // keep their order the same as in the ordered set.
                final int elementIndex = index.getAndIncrement();
                result.elements.add(component);

                graph.forTypes(component, new AsyncProcedure<Set<Resource>>() {

                    @Override
                    public void exception(AsyncReadGraph graph, Throwable t) {
                        if (errorHandler != null)
                            errorHandler.error(t.getMessage(), t);
                    }

                    @Override
                    public void execute(AsyncReadGraph graph, Set<Resource> types) {
                        if (types.contains(DIA.Connection)) {
                            if (types.contains(DIA.RouteGraphConnection)) {
                                graph.asyncRequest(new RouteGraphConnectionPartRequest(errorHandler, DIA, component),
                                        new ProcedureAdapter<RouteGraphConnectionPartData>() {
                                    @Override
                                    public void execute(RouteGraphConnectionPartData partData) {
                                        synchronized (result) {
                                            for (EdgeResource link : partData.links) {
                                                result.routeLinks.add(link);
                                                result.partToConnection.put(link, component);
                                                result.connectionToParts.add(component, link);
                                            }
                                            for (Resource line : partData.routeLines) {
                                                result.routeLines.add(line);
                                                result.connectionToParts.add(component, line);
                                                result.partToConnection.put(line, component);
                                            }
                                            for (Resource point : partData.routePoints) {
                                                result.routePoints.add(point);
                                                result.connectionToParts.add(component, point);
                                                result.partToConnection.put(point, component);
                                            }
                                        }
                                    }
                                });

                                synchronized (result.routeGraphConnectionSet) {
                                    result.routeGraphConnectionSet.add(component);
                                }
                            } else {
                                graph.asyncRequest(new ConnectionPartRequest(errorHandler, DIA, component),
                                        new ProcedureAdapter<ConnectionPartData>() {
                                    @Override
                                    public void execute(ConnectionPartData partData) {
                                        synchronized (result) {
                                            for (EdgeResource er : partData.edges) {
                                                result.connectionSegments.add(er);
                                                result.partToConnection.put(er, component);
                                                result.connectionToParts.add(component, er);
                                            }
                                            for (Resource bp : partData.branchPoints) {
                                                result.branchPoints.add(bp);
                                                result.connectionToParts.add(component, bp);
                                                result.partToConnection.put(bp, component);
                                            }
                                        }
                                    }
                                });

                                synchronized (result.connectionSet) {
                                    result.connectionSet.add(component);
                                }
                            }
                        }
                        else if (types.contains(DIA.Element)) {
                            synchronized (result.nodeSet) {
                                result.nodeSet.add(component);
                            }
                        }
                        else {
                            synchronized (unrecognizedElementIndices) {
                                // Unrecognized element, mark it to be
                                // removed after everything is processed.
                                unrecognizedElementIndices.add(elementIndex);
                            }
                        }
                    }

                });

            }

            @Override
            public void finished(AsyncReadGraph graph) {
                // Remove elements that were not recognized in descending order.
                unrecognizedElementIndices.sort();
                unrecognizedElementIndices.forEachDescending(new TIntProcedure() {
                    @Override
                    public boolean execute(int index) {
                        result.elements.remove(index);
                        return true;
                    }
                });

                // Help successive request executions by remembering the previous
                // element count. This will relieve some ArrayList reallocation
                // strain down the road.
                previousElementCount = result.elements.size();
            }

            @Override
            public void exception(AsyncReadGraph graph, Throwable t) {
                if (errorHandler != null)
                    errorHandler.error(t.getMessage(), t);
            }
        });

        return result;
    }
}