/*
 * Copyright (c) 2017 Goldman Sachs.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * and Eclipse Distribution License v. 1.0 which accompany this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 */

package org.eclipse.collections.impl.stack.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.EmptyStackException;

import org.eclipse.collections.api.ByteIterable;
import org.eclipse.collections.api.LazyByteIterable;
import org.eclipse.collections.api.bag.primitive.MutableByteBag;
import org.eclipse.collections.api.block.function.primitive.ByteToObjectFunction;
import org.eclipse.collections.api.block.function.primitive.ObjectByteIntToObjectFunction;
import org.eclipse.collections.api.block.function.primitive.ObjectByteToObjectFunction;
import org.eclipse.collections.api.block.predicate.primitive.BytePredicate;
import org.eclipse.collections.api.block.procedure.primitive.ByteIntProcedure;
import org.eclipse.collections.api.block.procedure.primitive.ByteProcedure;
import org.eclipse.collections.api.iterator.ByteIterator;
import org.eclipse.collections.api.list.primitive.ByteList;
import org.eclipse.collections.api.list.primitive.MutableByteList;
import org.eclipse.collections.api.set.primitive.MutableByteSet;
import org.eclipse.collections.api.stack.MutableStack;
import org.eclipse.collections.api.stack.primitive.ByteStack;
import org.eclipse.collections.api.stack.primitive.ImmutableByteStack;
import org.eclipse.collections.api.stack.primitive.MutableByteStack;
import org.eclipse.collections.impl.bag.mutable.primitive.ByteHashBag;
import org.eclipse.collections.impl.factory.primitive.ByteStacks;
import org.eclipse.collections.impl.iterator.UnmodifiableByteIterator;
import org.eclipse.collections.impl.lazy.primitive.LazyByteIterableAdapter;
import org.eclipse.collections.impl.list.mutable.primitive.ByteArrayList;
import org.eclipse.collections.impl.set.mutable.primitive.ByteHashSet;
import org.eclipse.collections.impl.stack.primitive.AbstractByteStack;
import org.eclipse.collections.impl.stack.mutable.ArrayStack;
import net.jcip.annotations.NotThreadSafe;

/**
 * ByteArrayStack is similar to {@link ArrayStack}, and is memory-optimized for byte primitives.
 * This file was automatically generated from template file primitiveArrayStack.stg.
 */
@NotThreadSafe
public final class ByteArrayStack extends AbstractByteStack
        implements MutableByteStack, Externalizable
{
    private static final long serialVersionUID = 1L;

    private transient ByteArrayList delegate;

    public ByteArrayStack()
    {
        this.delegate = new ByteArrayList();
    }

    private ByteArrayStack(int size)
    {
        this.delegate = new ByteArrayList(size);
    }

    private ByteArrayStack(byte... items)
    {
        this.delegate = new ByteArrayList(items);
    }

    public static ByteArrayStack newStackFromTopToBottom(byte... items)
    {
        ByteArrayStack stack = new ByteArrayStack(items.length);
        for (int i = items.length - 1; i >= 0; i--)
        {
            stack.push(items[i]);
        }
        return stack;
    }

    public static ByteArrayStack newStackWith(byte... items)
    {
        return new ByteArrayStack(items);
    }

    public static ByteArrayStack newStack(ByteIterable items)
    {
        ByteArrayStack stack = new ByteArrayStack(items.size());
        stack.delegate = ByteArrayList.newList(items);
        return stack;
    }

    public static ByteArrayStack newStackFromTopToBottom(ByteIterable items)
    {
        ByteArrayStack stack = new ByteArrayStack(items.size());
        stack.delegate = ByteArrayList.newList(items).reverseThis();
        return stack;
    }

    @Override
    protected ByteArrayList getDelegate()
    {
        return this.delegate;
    }

    public void push(byte item)
    {
        this.delegate.add(item);
    }

    public byte pop()
    {
        this.checkEmptyStack();
        return this.delegate.removeAtIndex(this.delegate.size() - 1);
    }

    public ByteList pop(int count)
    {
        this.checkPositiveValueForCount(count);
        this.checkSizeLessThanCount(count);
        if (count == 0)
        {
            return new ByteArrayList(0);
        }
        MutableByteList subList = new ByteArrayList(count);
        while (count > 0)
        {
            subList.add(this.pop());
            count--;
        }
        return subList;
    }

    public MutableByteStack select(BytePredicate predicate)
    {
        return newStackFromTopToBottom(this.delegate.asReversed().select(predicate));
    }

    public MutableByteStack reject(BytePredicate predicate)
    {
        return newStackFromTopToBottom(this.delegate.asReversed().reject(predicate));
    }

    public <V> MutableStack<V> collect(ByteToObjectFunction<? extends V> function)
    {
        return ArrayStack.newStackFromTopToBottom(this.delegate.asReversed().collect(function));
    }

    public void clear()
    {
        this.delegate.clear();
    }

    public MutableByteList toSortedList()
    {
        return ByteArrayList.newList(this).sortThis();
    }

    public MutableByteStack asUnmodifiable()
    {
        return new UnmodifiableByteStack(this);
    }

    public MutableByteStack asSynchronized()
    {
        return new SynchronizedByteStack(this);
    }

    public ImmutableByteStack toImmutable()
    {
        return ByteStacks.immutable.withAll(this.delegate);
    }

    public void writeExternal(ObjectOutput out) throws IOException
    {
        out.writeInt(this.size());
        ByteIterator iterator = this.delegate.asReversed().byteIterator();
        while (iterator.hasNext())
        {
            byte each = iterator.next();
            out.writeByte(each);
        }
    }

    public void readExternal(ObjectInput in) throws IOException
    {
        int size = in.readInt();
        byte[] array = new byte[size];
        for (int i = size - 1; i >= 0; i--)
        {
            array[i] = in.readByte();
        }
        this.delegate = ByteArrayList.newListWith(array);
    }
}
