package org.simantics.fileimport;

import java.nio.file.Path;
import java.util.Collection;
import java.util.Optional;

import org.simantics.Simantics;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.RuntimeDatabaseException;
import org.simantics.db.layer0.util.RemoverUtil;
import org.simantics.db.request.Read;
import org.simantics.db.service.SerialisationSupport;
import org.simantics.layer0.Layer0;

/**
 * Most of the implementations should extend this class which handles the storing of
 * the identifier of the imported entity and the removing of the entity
 * 
 * @author Jani Simomaa
 *
 */
public abstract class SimanticsResourceFileImport implements IGenericFileImport {

    @Override
    final public Optional<String> perform(Path file) throws Exception {
        
        Path dropins = Activator.getDropinsFolder();
        
        Path parts;
        if (file.toAbsolutePath().toString().startsWith(dropins.toAbsolutePath().toString())) {
        	parts = dropins.relativize(file);
        } else {
        	parts = file.getFileName();
        }
        
        Resource parent = resolveParent(null, parts);
        if (parent == null)
            return Optional.empty();
        Optional<Resource> imported = perform(parent, file); 
        if (imported.isPresent()) {
            return Optional.of(serialize(imported.get()));
        } else {
            return Optional.empty();
        }
    }
    
    /**
     * Performs the import for the given file
     * 
     * @param parent Resource parent of the imported entity in Simantics database
     * @param file Path file location of file
     * @return Optional Resource of the imported entity in Simantics database
     * @throws Exception
     */
    
    @Override
    public void remove(String resourceId) throws Exception {
        Optional<Resource> resource = deserialize(resourceId);
        resource.ifPresent(res -> {
            try {
                Simantics.sync(new WriteRequest() {

                    @Override
                    public void perform(WriteGraph graph) throws DatabaseException {
                        RemoverUtil.remove(graph, resource.get());
                    }
                });
            } catch (Exception e) {
                throw new RuntimeDatabaseException(e);
            }
        });
    }

    public String serialize(Resource resource) {
        return Long.toString(resource.getResourceId());
    }

    public Optional<Resource> deserialize(String serialized) throws Exception {
        long resourceId = Long.valueOf(serialized);

        Resource resource = Simantics.getSession().syncRequest(new Read<Resource>() {

            @Override
            public Resource perform(ReadGraph graph) throws DatabaseException {
                SerialisationSupport support = graph.getService(SerialisationSupport.class);
                Resource resource = support.getResource(resourceId);
                return resource;
            }
        });
        return Optional.ofNullable(resource);
    }
    
    private static Resource resolveParent(Resource parent, Path name) {
        if (name.getParent() == null) {
            return Simantics.getProjectResource();
        } else {
            name = name.getParent();
            parent = resolveParent(parent, name);
        }
        final Resource newParent = parent;
        final String folderName = name.getFileName().toString();

        try {
            return Simantics.getSession().syncRequest(new UniqueRead<Resource>() {

                @Override
                public Resource perform(ReadGraph graph) throws DatabaseException {
                    Layer0 L0 = Layer0.getInstance(graph);
                    Collection<Resource> libraries = graph.sync(new ObjectsWithType(newParent, L0.ConsistsOf, L0.Library));
                    for (Resource library : libraries) {
                        String libraryName = graph.getRelatedValue2(library, L0.HasName, Bindings.STRING);
                        if (libraryName.equals(folderName)) {
                            return library;
                        }
                    }
                    return null;
                }
            });
        } catch (DatabaseException e) {
            e.printStackTrace();
            return null;
        }
    }
    
    public abstract Resource defaultParentResource();

}
