package org.simantics.document.linking.report.templates;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.document.DocumentResource;
import org.simantics.document.linking.ontology.DocumentLink;
import org.simantics.document.linking.report.Document;
import org.simantics.document.linking.report.DocumentTitlePage;
import org.simantics.document.linking.report.RowContentProvider;
import org.simantics.document.linking.report.Table;
import org.simantics.document.linking.report.TableOfContents;
import org.simantics.document.linking.report.TextItem;
import org.simantics.document.linking.utils.SourceLinkUtil;

/**
 *  ? Configuration folder 1
 *     Diagram 1
 *        o Link1: Document information
 *        o Link2: Document information
 *        o Link3: Document information
 *        o Module 1
 *            ? Link4: Document information
 *            ? Link5: Document information
 *            ? Attribute 1
 *                ? Link6: Document information
 *            ? Attribute 2
 *                ? Link7: Document information
 *                ? Link8: Document information
 *        o Module 2
 *            ? Link9: Document information
 *            
 * @author Marko Luukkainen <marko.luukkainen@vtt.fi>
 *
 */
public class CompleteStructureWriter extends IndexQueryReport {
		ReadGraph graph;
		Resource model;
		Map<Object, Object> context;
		DocumentLink sl;
		DocumentResource doc;
		ResourceNameComparator parentComparator;
		Comparator<Resource> hierarchyComparator;
		
		@Override
		public String getName() {
			return "Complete Structure";
		}
		
		@Override
		public void sort(List<Resource> list) throws DatabaseException{
			Collections.sort(list, parentComparator);
		}
		
		@Override
		public void start(final ReadGraph graph, final Resource model, Document writer, Map<Object, Object> context) throws Exception{
			super.start(graph, model, writer, context);
			this.context = context;
			DocumentTitlePage titlePage = writer.newElement(DocumentTitlePage.class);
			titlePage.writeTitle(graph, context);
			writer.newElement(TableOfContents.class);
			Table table = writer.newElement(Table.class,Document.TOC);
			
			table.addColumn("Name", 0.2);
			table.addColumn("Attribute", 0.2);
			table.addColumn("Value", 0.15);
			table.addColumn("Document", 0.2);
			table.addColumn("Comment", 0.25);
			
			//lineWriter.nextPage();
			
			this.graph = graph;
			this.sl = DocumentLink.getInstance(graph);
			this.doc = DocumentResource.getInstance(graph);
			this.model = model;
			
			
			
			parentComparator = new SourceParentHierarchyWithPropComparator(graph,model);
			hierarchyComparator = new SourceParentDiagramComparator(graph,model);
				
			clearProviders();
			addLineProvider(new HierarchyProvider());
			addCellProvider(new LinkCellProvider());
			
		}
		
		private class HierarchyProvider implements RowContentProvider<Resource> {
			
			@Override
			public void setText(Document writer, Resource previous, Resource current, Resource next, TextItem[] row) throws Exception {
				boolean writeHierarchy = false;
				if (previous == null)
					writeHierarchy = true;
				else  {
					if (hierarchyComparator.compare(previous, current) != 0)
						writeHierarchy = true;
				}
				if (writeHierarchy) {
					Resource obj = graph.getPossibleObject(current, sl.hasSource_Inverse);
					Table table = writer.getCurrentElement(Table.class);

					if (writer.getCurrentLine() > 2) {
						writer.nextPage();
						table = writer.nextElement(Table.class,Document.TOC);
					}
					
					if (obj != null) {
						String text = "";
						List<Resource> path = SourceLinkUtil.getDiagramPath(graph, model, obj);
						
						if (path == null) {
							path = parentComparator.getPath(obj);
							for (int i = 0 ; i < path.size()-1; i++) {
								Resource r = path.get(i);
								text += parentComparator.getText(r);
								if (i < path.size()-2)
									text += "/";
							}
						} else {
							for (int i = 0 ; i < path.size(); i++) {
								Resource r = path.get(i);
								text += parentComparator.getText(r);
								if (i < path.size()-1)
									text += "/";
							}
						}
						//row[0] = text;
						table.setTitle(text);
						
						TableOfContents toc = writer.getCurrentElement(TableOfContents.class);
						toc.addTocElement(text, table);
					} else {
						//row[0] =  "Hierarchy missing";
						table.setTitle("Hierarchy missing");
					}
				}
			}
		}
		
		private class LinkCellProvider implements RowContentProvider<Resource> {
			@Override
			public void setText(Document writer, Resource previous, Resource source, Resource next, TextItem[] text)
					throws Exception {
				Resource holder = graph.getSingleObject(source, sl.hasSource_Inverse);
				String holderName = NameUtils.getSafeLabel(graph, holder);
				if (holderName.length() == 0)
					holderName = NameUtils.getSafeName(graph, holder);
				if (previous == null || !graph.getSingleObject(previous, sl.hasSource_Inverse).equals(graph.getSingleObject(source, sl.hasSource_Inverse))) {
					text[0] = writer.newItem(TextItem.class);
					text[0].setText(holderName);
				}
				if (graph.isInstanceOf(source, sl.FunctionalSource)) {
					Resource relation = graph.getPossibleObject(source, sl.consernsRelation);
					text[1] = writer.newItem(TextItem.class);
					if (relation != null) {
						String relationName = NameUtils.getSafeLabel(graph, relation);
						if (relationName.length() == 0)
							relationName = NameUtils.getSafeName(graph, relation);
						Object value = graph.getPossibleRelatedValue(holder, relation);
						text[1].setText(relationName);
						if (value != null) {
							text[2] = writer.newItem(TextItem.class);
							text[2].setText(SourceLinkUtil.getValueString(value));
						}
					} else {
						text[1].setText("Error in property reference");
					}
				}
				Resource document = SourceLinkUtil.getReferredDocument(graph, source);
				if (document != null) {
					text[3] = getDocumentItem(document);
				} else {
					text[3] = getNonExistingDocumentItem();
				}
				
				String comment = graph.getPossibleRelatedValue(source, sl.hasSourceComment);
				if (comment != null) {
					text[4] = writer.newItem(TextItem.class);
					text[4].setText(comment);
				}

			}
		}
	}