package org.simantics.document.linking.report.pdf;

import java.awt.Font;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import com.lowagie.text.Document;
import com.lowagie.text.PageSize;
import com.lowagie.text.Rectangle;
import com.lowagie.text.pdf.PdfCopy;
import com.lowagie.text.pdf.PdfImportedPage;
import com.lowagie.text.pdf.PdfReader;

public class PDFPageStream {
	boolean modifiable = false;
	Rectangle pageSize = PageSize.A4;
	Font defaultFont;
	List<PDFPage> pages = new ArrayList<PDFPage>();
	
	int contentWidth;
	int contentHeight;
	
	int marginLeft = 20;
    int marginRight = 20;
    int marginTop = 10;
    int marginBottom = 45;

	int currentPage = 0;
	
	public PDFPageStream() {
		calculateContent();
	}
	
	public PDFPageStream(Rectangle pageSize) {
		this.pageSize = pageSize;
		calculateContent();
	}
	
	public PDFPageStream(boolean modifiable) {
		this.modifiable = modifiable;
		calculateContent();
	}
	
	public PDFPageStream(Rectangle pageSize, boolean modifiable) {
		this.pageSize = pageSize;
		this.modifiable = modifiable;
		calculateContent();
	}
	
	private void calculateContent() {
		contentWidth = (int)getPageSize().getWidth() - marginLeft - marginRight;
        contentHeight = (int)getPageSize().getHeight() - marginTop - marginBottom;
	}
	

	public Rectangle getPageSize() {
		return pageSize;
	}
	
	public Font getDefaultFont() {
		return defaultFont;
	}
	
	public void setDefaultFont(Font defaultFont) {
		this.defaultFont = defaultFont;
	}
	
	public List<PDFPage> getPages() {
		return pages;
	}
	
	public void nextPage() throws Exception{
		endPage();
		
		PDFPage page = new PDFPage(this);
		pages.add(page);
		currentPage++;
		
	}

	private void endPage() {
		if (pages.size() > 0) {
			PDFPage lastPage = pages.get(pages.size()-1);
			if (!modifiable && lastPage.isOpen())
				lastPage.close();
		}
	}
	
	
	
	public int getAvailableLines() {
		return getCurrentPage().availableLines;
	}
	
//	public int getTotalLines() {
//		return totalLines;
//	}
	
	public int getCurrentLine() {
		return getCurrentPage().currentLine;
	}
	
	public int getContentWidth() {
		return contentWidth;
	}
	
	public int getContentHeight() {
		return contentHeight;
	}
	
	protected void checkNextPage() throws Exception{
		PDFPage page = getCurrentPage();
		if (contentHeight - page.currentPixel - page.getLineHeight() <= 0)
			nextPage();
	}
	
	public int getCurrentPageIndex() {
		return currentPage;
	}
	
	public PDFPage getCurrentPage() {
		return pages.get(currentPage-1);
	}
	
	public void copy(PdfCopy pdfCopy) throws IOException{
		for (PDFPage page : pages) {
			if (page.isOpen())
				page.close();
			PdfReader reader = new PdfReader(page.tempFile.getAbsolutePath());
			try {
		        for (int i = 1; i <= reader.getNumberOfPages(); i++) {
		        	PdfImportedPage ipage = pdfCopy.getImportedPage(reader, i);
		            pdfCopy.addPage(ipage);
		        }
			} catch (Exception e) {
				e.printStackTrace();
	        } finally {
	        	reader.close();
	        }
		}
	}
	
	public void close(String filename) throws Exception{
		if (pages.size() == 0) {
			return;
		}
		try {
			endPage();
			Document document = new Document(pageSize);
			PdfCopy pdfCopy = new PdfCopy(document, new FileOutputStream(filename));
			document.open();
			copy(pdfCopy);
			document.close();
			pdfCopy.close();
		} finally {
			close();
		}
	}
	
	public void close() throws Exception{
		for (PDFPage page : pages) {
				page.tempFile.delete();
		}
		pages.clear();
		defaultFont = null;
	}
}
