package org.simantics.document.linking.report.pdf;

import java.awt.Font;
import java.util.HashMap;
import java.util.Map;

import org.simantics.document.linking.report.Document;
import org.simantics.document.linking.report.DocumentElement;
import org.simantics.document.linking.report.DocumentItem;
import org.simantics.document.linking.report.DocumentTitlePage;
import org.simantics.document.linking.report.Table;
import org.simantics.document.linking.report.TableOfContents;
import org.simantics.document.linking.report.TextItem;
import org.simantics.document.linking.report.URLItem;
import org.simantics.document.linking.report.base.TextItemImpl;
import org.simantics.document.linking.report.base.URLItemImpl;
import org.simantics.utils.datastructures.Arrays;


/**
 * PDF format reporter.
 * 
 * @author Marko Luukkainen <marko.luukkainen@vtt.fi>
 *
 */
public class PDFDocument implements Document {
	private String filename;
	
	Map<TextSize, Font> fonts = new HashMap<TextSize, Font>();
	
	PDFPageStream contentStream;
	PDFTable currentTable;
	PDFTitlePage titlePage;
	PDFTocElement toc;
	
	public PDFDocument(String filename) throws Exception {
		this.filename = filename;
		contentStream = new PDFPageStream();
		defaultFonts();
		contentStream.setDefaultFont(fonts.get(TextSize.SMALL));
	}
	
	public PDFDocument(String filename, Font font) throws Exception {
		this.filename = filename;
		contentStream = new PDFPageStream();
		defaultFonts();
		fonts.put(TextSize.SMALL, font);
	}
	
	public PDFDocument(String filename, Font font, Font headerFont) throws Exception {
		this.filename = filename;
		contentStream = new PDFPageStream();
		defaultFonts();
		fonts.put(TextSize.SMALL, font);
		fonts.put(TextSize.MEDIUM, headerFont);
	}
	
	private void defaultFonts() {
		fonts.put(TextSize.TINY, new Font("Arial", Font.PLAIN, 6));
		fonts.put(TextSize.SMALL, new Font("Arial", Font.PLAIN, 8));
		fonts.put(TextSize.MEDIUM, new Font("Arial", Font.PLAIN, 10));
		fonts.put(TextSize.LARGE, new Font("Arial", Font.PLAIN, 16));
		fonts.put(TextSize.HUGE, new Font("Arial", Font.BOLD, 24));
		contentStream.setDefaultFont(fonts.get(TextSize.SMALL));
	}
	
	Font getFont(TextSize size) {
		return fonts.get(size);
	}
	
	@Override
	public int getAvailableLines() {
		return contentStream.getAvailableLines();
	}
	
	@Override
	public int getCurrentLine() {
		return contentStream.getCurrentLine();
	}
	
	public void close() throws Exception{
		if (toc != null) {
			// hackety hack.
			toc.create(contentStream, contentStream.getPages().get(0));
		}
		contentStream.close(filename);
	}
	
	/* (non-Javadoc)
	 * @see org.simantics.document.linking.report.Document#nextPage()
	 */
	@Override
	public void nextPage() throws Exception{
		contentStream.nextPage();	
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public <T extends DocumentElement> T newElement(Class<T> cls, String...options)	throws Exception {
		if (cls == Table.class) {
			return (T)newTable();
		} else if (cls == TableOfContents.class) {
			return (T)getOrCreateToc();
		} else if (cls == DocumentTitlePage.class) {
			if (titlePage != null)
				throw new Exception("Report may contain only one title page");
			titlePage =  new PDFTitlePage(this);
			return (T)titlePage;
		}
		return null;
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public <T extends DocumentElement> T nextElement(Class<T> cls, String...options) throws Exception {
		if (cls == Table.class) {
			return (T)nextTable(Arrays.contains(options, Document.TOC));
		} else if (cls == TableOfContents.class) {
			return (T)getOrCreateToc();
		} 
		return null;
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public <T extends DocumentElement> T getCurrentElement(Class<T> cls) {
		if (cls == Table.class) {
			return (T)getCurrentTable();
		} else if (cls == TableOfContents.class) {
			return (T)toc;
		} else if (cls == DocumentTitlePage.class) {
			return (T)titlePage;
		}
		return null;
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public <T extends DocumentItem> T newItem(Class<T> cls, String... options)
			throws Exception {
		if (cls == TextItem.class)
			return (T)new TextItemImpl();
		if (cls == URLItem.class)
			return (T)new URLItemImpl();
		return null;
	}	
	
	public Table nextTable(boolean id) throws Exception {
		if (currentTable!= null) {
			closeTable();
			currentTable = new PDFTable(currentTable);
			return currentTable;	
		} else {
			return newTable();
		}

	}
	
	private void closeTable() throws Exception {
		if (currentTable != null && currentTable.currentLine > 0) {
			currentTable.setLinesVisible(false);
			currentTable.setHeaderVisible(false);
			currentTable.writeLine("");
		}
	}
	
	
	public Table newTable() throws Exception {
		closeTable();
		currentTable = new PDFTable(this,contentStream);
		return currentTable;
	}
	
	
	Table getCurrentTable() {
		return currentTable;
	}
	
	TableOfContents getOrCreateToc() throws Exception{
		if (toc == null)
			toc = new PDFTocElement(this);
		return toc;
	}
	
	


}
