package org.simantics.document.linking.report.evaluator;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.NamedResource;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.request.Read;
import org.simantics.document.linking.ontology.DocumentLink;
import org.simantics.document.linking.report.templates.CustomizableContentProvider;
import org.simantics.document.linking.report.templates.custom.CustomizableContent;
import org.simantics.document.linking.report.templates.custom.EvaluatorCustomizableContent;
import org.simantics.layer0.Layer0;
import org.simantics.objmap.graph.IMapping;
import org.simantics.objmap.graph.Mappings;
import org.simantics.objmap.graph.schema.IMappingSchema;
import org.simantics.objmap.graph.schema.MappingSchemas;
import org.simantics.objmap.graph.schema.SimpleSchema;

public class DBUtil {
	
	/**
	 * Loads single evaluator tree from graph.
	 * @param graph
	 * @param res
	 * @return
	 * @throws DatabaseException
	 */
	public static EvaluatorItem load(ReadGraph graph, Resource res) throws DatabaseException{
		IMappingSchema<Resource, Object> schema = getSchema(graph); 
		IMapping<Resource,Object> mapping =  Mappings.createWithoutListening(schema);
		EvaluatorItem item = (EvaluatorItem)mapping.map(graph, res);
		return item;
	}
	
	/**
	 * Saves a single evaluator tree to graph.
	 * @param graph
	 * @param item
	 * @return
	 * @throws DatabaseException
	 */
	public static Resource save(WriteGraph graph, EvaluatorItem item) throws DatabaseException{
		IMappingSchema<Resource, Object> schema = getSchema(graph); 
		IMapping<Resource,Object> mapping =  Mappings.createWithoutListening(schema);
		Resource resource =  mapping.inverseMap(graph, item);
		return resource;
	}
	
	/**
	 * Creates mapping schema for Evaluator tree.
	 * @param g
	 * @return
	 * @throws DatabaseException
	 */
	public static IMappingSchema<Resource,Object> getSchema(ReadGraph g) throws DatabaseException{
		try {
			SimpleSchema schema = new SimpleSchema();
			schema.addLinkType(MappingSchemas.fromAnnotations(g, AlignmentHint.class));
			schema.addLinkType(MappingSchemas.fromAnnotations(g, And.class));
			schema.addLinkType(MappingSchemas.fromAnnotations(g, Constant.class));
			schema.addLinkType(MappingSchemas.fromAnnotations(g, Date.class));
			schema.addLinkType(MappingSchemas.fromAnnotations(g, EvaluatorRoot.class));
			schema.addLinkType(MappingSchemas.fromAnnotations(g, If.class));
			schema.addLinkType(MappingSchemas.fromAnnotations(g, Lines.class));
			schema.addLinkType(MappingSchemas.fromAnnotations(g, Or.class));
			schema.addLinkType(MappingSchemas.fromAnnotations(g, TextSizeHint.class));
			schema.addLinkType(MappingSchemas.fromAnnotations(g, Variable.class));
			schema.addLinkType(MappingSchemas.fromAnnotations(g, Path.class));
			
			return schema;
		} catch (IllegalAccessException e) {
			throw new DatabaseException(e);
		} catch (InstantiationException e) {
			throw new DatabaseException(e);
		}
	}
	
	/**
	 * Saves evaluator tree configurations of content provider.
	 * @param graph
	 * @param provider
	 * @param template
	 * @throws DatabaseException
	 */
	public static void save(WriteGraph graph, CustomizableContentProvider provider, Resource template) throws DatabaseException{
		Layer0 l0 = Layer0.getInstance(graph);
		DocumentLink sl = DocumentLink.getInstance(graph);
		
		Map<String, Resource> existingEvaluatorTemplates = new HashMap<String, Resource>();
		for (Resource r : graph.getObjects(template, l0.ConsistsOf)) {
			if (graph.isInstanceOf(r, sl.EvaluatorTree_Template)) {
				String id = graph.getRelatedValue(r, l0.HasName);
				existingEvaluatorTemplates.put(id, r);
			}
		}
		
		for (String id : provider.getContentIds()) {
			CustomizableContent content = provider.getContent(id);
			if (content instanceof EvaluatorCustomizableContent) {
				EvaluatorCustomizableContent ecc = (EvaluatorCustomizableContent)content;
				Resource item = save(graph, ecc.getItem());
				
				Resource treeTemplate = existingEvaluatorTemplates.get(id);
				if (treeTemplate == null) {
					treeTemplate = graph.newResource();
					graph.claim(treeTemplate, l0.InstanceOf, sl.EvaluatorTree_Template);
					graph.claimLiteral(treeTemplate, l0.HasName, id);
					graph.claim(template, l0.ConsistsOf, treeTemplate);
				} else {
					graph.deny(treeTemplate,l0.ConsistsOf);
				}
				graph.claim(treeTemplate, l0.ConsistsOf, item);
			}
		}
	}
	
	/**
	 * Loads evaluator tree configurations from a graph and applies them to content provider.
	 * @param graph
	 * @param template
	 * @param provider
	 * @throws DatabaseException
	 */
	public static void load(ReadGraph graph, Resource template, CustomizableContentProvider provider) throws DatabaseException {
		Layer0 l0 = Layer0.getInstance(graph);
		DocumentLink sl = DocumentLink.getInstance(graph);
	
		Map<String, Resource> existingEvaluatorTemplates = new HashMap<String, Resource>();
		for (Resource r : graph.getObjects(template, l0.ConsistsOf)) {
			if (graph.isInstanceOf(r, sl.EvaluatorTree_Template)) {
				String id = graph.getRelatedValue(r, l0.HasName);
				existingEvaluatorTemplates.put(id, r);
			}
		}
		
		for (String id : provider.getContentIds()) {
			Resource treeTemplate = existingEvaluatorTemplates.get(id);
			if (treeTemplate == null)
				continue;
			
			Resource itemRes = graph.getSingleObject(treeTemplate, l0.ConsistsOf);
			EvaluatorItem item = load(graph, itemRes);
			EvaluatorCustomizableContent ecc = new EvaluatorCustomizableContent(provider.getContent(id).getCustomizationDescription());
			ecc.setItem(item);
			provider.setContent(id, ecc);
			 
		}	
		
	}
	
	public static List<NamedResource> getTemplates(final Resource library) throws DatabaseException{
		if (library == null)
			throw new IllegalArgumentException("Library cannot be null");
		List<NamedResource> templates = Simantics.getSession().syncRequest(new Read<List<NamedResource>>() {
		@Override
		public List<NamedResource> perform(ReadGraph graph) throws DatabaseException{
				return getTemplates(graph, library);
			}
		});
		return templates;
	}
	
	public static List<NamedResource> getTemplates(ReadGraph graph, final Resource library) throws DatabaseException{
		Layer0 l0 = Layer0.getInstance(graph);
		DocumentLink sl = DocumentLink.getInstance(graph);
		Collection<Resource> templates = graph.syncRequest(new ObjectsWithType(library, l0.ConsistsOf, sl.ReportTemplate));
		List<NamedResource> result = new ArrayList<NamedResource>(templates.size());
		for (Resource template : templates) {
			String name = graph.getRelatedValue(template, l0.HasName);
			result.add(new NamedResource(name, template));
		}

		Collections.sort(result);
		return result;
	}
	
	public static void save(final Resource library, final String name,final  CustomizableContentProvider provider) throws DatabaseException{
		Simantics.getSession().syncRequest(new WriteRequest() {
			
			@Override
			public void perform(WriteGraph graph) throws DatabaseException {
				final List<NamedResource> templates = DBUtil.getTemplates(graph,library);
				Resource template = null;
				for (NamedResource nr : templates) {
					if (name.equals(nr.getName())) {
						template = nr.getResource();
						break;
					}
				}
				if (template == null) {
					Layer0 l0 = Layer0.getInstance(graph);
					DocumentLink sl = DocumentLink.getInstance(graph);
					template = graph.newResource();
					graph.claim(template, l0.InstanceOf, sl.ReportTemplate);
					graph.claim(library, l0.ConsistsOf, template);
					graph.claimLiteral(template, l0.HasName, name);
				}
				DBUtil.save(graph, provider, template);
			}
		});
	}

}
