package org.simantics.scl.compiler.module;

import org.simantics.scl.compiler.elaboration.expressions.EVar;
import org.simantics.scl.compiler.internal.parsing.Symbol;
import org.simantics.scl.compiler.internal.parsing.declarations.DeclarationAst;


public class ImportDeclaration extends DeclarationAst {
    public static final ImportSpec DEFAULT_SPEC = new ImportSpec(true, new EVar[0]);
    
    public static final ImportDeclaration[] EMPTY_ARRAY = new ImportDeclaration[0];

    public static final ImportDeclaration[] ONLY_BUILTINS =
            new ImportDeclaration[] { new ImportDeclaration("Builtin", "") };

    
    public final String moduleName;
    public final String localName;
    public final boolean reexport;
    public final ImportSpec spec;
    
    public static class ImportSpec extends Symbol {
        public final boolean hiding;
        public final EVar[] values;
        
        public ImportSpec(boolean hiding, EVar[] values) {
            this.hiding = hiding;
            this.values = values;
        }
        
        @Override
        public String toString() {
            if(hiding && values.length == 0)
                return "";
            StringBuilder b = new StringBuilder();
            if(hiding)
                b.append("hiding ");
            b.append('(');
            for(int i=0;i<values.length;++i) {
                if(i > 0)
                    b.append(',');
                b.append(values[i].name);
            }
            b.append(')');
            return b.toString();
        }
    }
    
    public ImportDeclaration(String moduleName, String localName, boolean reexport) {
        this.moduleName = moduleName;
        this.localName = localName;
        this.reexport = reexport;
        this.spec = DEFAULT_SPEC;
    }
    
    public ImportDeclaration(long location, String moduleName, String localName, boolean reexport, ImportSpec spec) {
        this(moduleName, localName, reexport);
        this.location = location;
    }
    
    public ImportDeclaration(String moduleName, String localName, boolean reexport, ImportSpec spec) {
        if(spec == null)
            throw new NullPointerException();
        this.moduleName = moduleName;
        this.localName = localName;
        this.reexport = reexport;
        this.spec = spec;
    }
    
    public ImportDeclaration(String moduleName, String localName) {
        this(moduleName, localName, false);
    }

    @Override
    public void toString(int indentation, StringBuilder b) {
        for(int i=0;i<indentation;++i) b.append("    ");
        if (reexport)
            b.append("include \"");
        else
            b.append("import \"");
        b.append(moduleName);
        b.append("\" as ");
        b.append(localName);
        
    }

    public String getSpecString() {
        return spec.toString();
    }
    
    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result
                + ((localName == null) ? 0 : localName.hashCode());
        result = prime * result
                + ((moduleName == null) ? 0 : moduleName.hashCode());
        result = prime * result + (reexport ? 1231 : 1237);
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        ImportDeclaration other = (ImportDeclaration) obj;
        if (localName == null) {
            if (other.localName != null)
                return false;
        } else if (!localName.equals(other.localName))
            return false;
        if (moduleName == null) {
            if (other.moduleName != null)
                return false;
        } else if (!moduleName.equals(other.moduleName))
            return false;
        if (reexport != other.reexport)
            return false;
        return true;
    }    
    
}
