/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import gnu.trove.procedure.TIntProcedure;

import java.util.concurrent.Semaphore;
import java.util.concurrent.atomic.AtomicInteger;

import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.impl.procedure.InternalProcedure;
import org.simantics.db.procedure.ListenerBase;

final public class Types extends UnaryQuery<InternalProcedure<IntSet>> {
    
//	public ArrayList<InternalProcedure<IntSet>> procs;
	
    private Types(final int resource) {
        super(resource);
    }
    
    final static Types entry(final QueryProcessor provider, final int r) {
        return (Types)provider.typesMap.get(r);
    }
    
   final static void runner(ReadGraphImpl graph, final int r, final QueryProcessor provider, Types cached, final CacheEntry parent, final ListenerBase listener, final InternalProcedure<IntSet> procedure) {

    	Types entry = cached != null ? cached : (Types)provider.typesMap.get(r); 
        if(entry == null) {
        	
        	entry = new Types(r);
        	entry.setPending();
        	entry.clearResult(provider.querySupport);
        	entry.putEntry(provider);
        	
            provider.performForEach(graph, entry, parent, listener, procedure);
            
        } else {

        	if(!entry.isReady()) {
        	    throw new IllegalStateException();
            }
            provider.performForEach(graph, entry, parent, listener, procedure);
        }
    	
    }
    
    final static IntSet runner2(ReadGraphImpl graph, final int r, final QueryProcessor provider, final CacheEntry parent) throws Throwable {

    	Types entry = (Types)provider.typesMap.get(r); 
        if(entry == null) {
        	
        	entry = new Types(r);
        	entry.setPending();
        	entry.clearResult(provider.querySupport);
        	entry.putEntry(provider);
        	
            return (IntSet)provider.performForEach2(graph, entry, parent, null, null);
            
        } else {

        	if(!entry.isReady()) {
        	    throw new IllegalStateException();
            }
            return (IntSet)provider.performForEach2(graph, entry, parent, null, null);
            
        }
    	
    }
    
    final public static void queryEach(ReadGraphImpl graph, final int r, final QueryProcessor provider, final CacheEntry parent, final ListenerBase listener, final InternalProcedure<IntSet> procedure) {
        
    	final Types entry = (Types)provider.typesMap.get(r);
    	
    	if(parent == null && listener == null) {
        	if(entry != null && entry.isReady()) { 
        		entry.performFromCache(graph, provider, procedure);
        		return;
        	}
        }
    	
        runner(graph, r, provider, entry, parent, listener, procedure);
         
    }

    final public static IntSet queryEach2(ReadGraphImpl graph, final int r, final QueryProcessor provider, final CacheEntry parent) throws Throwable {
        
        if(parent == null) {
            Types entry = (Types)provider.typesMap.get(r);
            if(entry != null && entry.isReady()) {
                return (IntSet)entry.get(graph, provider, null);
            }
        }
        
        return runner2(graph, r, provider, parent);
         
    }
    
 	@Override
 	public UnaryQuery<InternalProcedure<IntSet>> getEntry(QueryProcessor provider) {
        return provider.typesMap.get(id);
 	}
 	
 	@Override
 	public void putEntry(QueryProcessor provider) {
        provider.typesMap.put(id, this);
 	}

 	@Override
 	final public void removeEntry(QueryProcessor provider) {
 		provider.typesMap.remove(id);
 	}

 	@Override
 	public Object computeForEach(final ReadGraphImpl graph, final QueryProcessor queryProvider, final InternalProcedure<IntSet> procedure, final boolean store) {

 		queryProvider.querySupport.ensureLoaded(graph, id);
 		int ret = queryProvider.querySupport.getSingleInstance(id);
 		if(ret > 0) {

 			TypeHierarchy.queryEach(graph, ret, queryProvider, store ? Types.this : null, null, new InternalProcedure<IntSet>() {

 				@Override
 				public void execute(ReadGraphImpl graph, IntSet types) {

 					addOrSet(graph, types, queryProvider);
 					procedure.execute(graph, types);

 				}

 				@Override
 				public void exception(ReadGraphImpl graph, Throwable t) {
 					procedure.exception(graph, t);
 				}

 			});

 			return getResult();

 		}
 		
 		final int instanceOf = queryProvider.getInstanceOf();
        final int inherits = queryProvider.getInherits();
        final int subrelationOf = queryProvider.getSubrelationOf();

        final IntSet result = new IntSet(queryProvider.querySupport);
        
        final TIntProcedure addToResult = new TIntProcedure() {
            @Override
            public boolean execute(int r) {
            	synchronized(result) {
            		result.add(r);
            	}
                return true;
            }
        };
        
        final AtomicInteger finishes = new AtomicInteger(0);
        
        SyncIntProcedure instanceOfProcedure = new SyncIntProcedure() {
            
            @Override
            public void run(ReadGraphImpl graph) {
            	
                if(finishes.addAndGet(1) == 3) {
                    if(store) addOrSet(graph, result, queryProvider);
                    procedure.execute(graph, result);
            	}
                
            }
            
            @Override
            public void execute(ReadGraphImpl graph, int i) {
                
            	synchronized(result) {
            		result.add(i);
            	}
                
                inc();

                SuperTypes.queryEach(graph, i, queryProvider, store ? Types.this : null, null, new InternalProcedure<IntSet>() {

                    @Override
                    public void execute(ReadGraphImpl graph, IntSet types) {
                        types.forEach(addToResult);
                        dec(graph);
                    }
    				
    				@Override
    				public void exception(ReadGraphImpl graph, Throwable t) {
    					procedure.exception(graph, t);
                        dec(graph);
    	            }

                });
                
            }

            @Override
            public void finished(ReadGraphImpl graph) {
                dec(graph);
            }
            
        }; 
        
        SyncIntProcedure inheritsProcedure = new SyncIntProcedure() {
            
            @Override
            public void run(ReadGraphImpl graph) {

            	int current = finishes.addAndGet(1);
            	if(current == 3) {
                    if(store) addOrSet(graph, result, queryProvider);
                    procedure.execute(graph, result);
            	}
                
            }
            
            @Override
            public void execute(ReadGraphImpl graph, int i) {
                
                inc();

                Types.queryEach(graph, i, queryProvider, store ? Types.this : null, null, new InternalProcedure<IntSet>() {

                    @Override
                    public void execute(ReadGraphImpl graph, IntSet types) {
                        types.forEach(addToResult);
                        dec(graph);
                    }
    				
    				@Override
    				public void exception(ReadGraphImpl graph, Throwable t) {
    					procedure.exception(graph, t);
                        dec(graph);
    	            }

                });
                
            }

            @Override
            public void finished(ReadGraphImpl graph) {

                dec(graph);

            }
            
        }; 

        SyncIntProcedure subrelationOfProcedure = new SyncIntProcedure() {

            @Override
            public void run(ReadGraphImpl graph) {

            	int current = finishes.addAndGet(1);
            	if(current == 3) {
                    if(store) addOrSet(graph, result, queryProvider);
                    procedure.execute(graph, result);
            	}
                
            }
            
            @Override
            public void execute(ReadGraphImpl graph, int i) {
                
                inc();
                
                Types.queryEach(graph, i, queryProvider, store ? Types.this : null, null, new InternalProcedure<IntSet>() {

                    @Override
                    public void execute(ReadGraphImpl graph, IntSet types) {

                        types.forEach(addToResult);
                        dec(graph);
                        
                    }
    				
    				@Override
    				public void exception(ReadGraphImpl graph, Throwable t) {
    					procedure.exception(graph, t);
                        dec(graph);
    	            }

                });
                
            }

            @Override
            public void finished(ReadGraphImpl graph) {
                
                dec(graph);

            }
            
        }; 
        
        queryProvider.querySupport.getObjects(graph, id, instanceOf, instanceOfProcedure);
        instanceOfProcedure.finished(graph);
        queryProvider.querySupport.getObjects(graph, id, inherits, inheritsProcedure);
        inheritsProcedure.finished(graph);
        queryProvider.querySupport.getObjects(graph, id, subrelationOf, subrelationOfProcedure);
        subrelationOfProcedure.finished(graph);
        
        return result;
        
    }
    
    @Override
    public String toString() {
    	return "Types[" + id + "]";
    }
    
    private void addOrSet(ReadGraphImpl graph, final IntSet value, QueryProcessor provider) {
    	
        assert(!isReady());

        synchronized(this) {
        
            value.trim();
            setResult(value);
            setReady();
        
        }
        
    }
    
    @Override
    final public Object performFromCache(ReadGraphImpl graph, QueryProcessor provider, InternalProcedure<IntSet> procedure) {
        
    	assert(isReady());
    	
    	if(handleException(graph, procedure)) return EXCEPTED;
    	
    	IntSet result = getResult();
    	
        procedure.execute(graph, result);
        
        return result;
        
    }
    
    @Override
    public void recompute(ReadGraphImpl graph, QueryProcessor provider) {
        
        final Semaphore s = new Semaphore(0);

        computeForEach(graph, provider, new InternalProcedure<IntSet>() {

        	@Override
        	public void execute(ReadGraphImpl graph, IntSet result) {
                s.release();
        	}

            @Override
            public void exception(ReadGraphImpl graph, Throwable t) {
            	s.release();
                new Error("Error in recompute.", t).printStackTrace();
            }

        }, true);

        while(!s.tryAcquire()) {
        	provider.resume(graph);
        }
        
    }

    @Override
    boolean isImmutable(ReadGraphImpl graph) {
    	return graph.processor.isImmutable(id);
    }
    
}
