/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import java.util.Collection;
import java.util.concurrent.Semaphore;

import org.simantics.databoard.Bindings;
import org.simantics.db.DevelopmentKeys;
import org.simantics.db.RelationInfo;
import org.simantics.db.Resource;
import org.simantics.db.common.exception.DebugException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ManyObjectsForFunctionalRelationException;
import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.impl.procedure.IntProcedureAdapter;
import org.simantics.db.impl.procedure.InternalProcedure;
import org.simantics.db.procedure.AsyncMultiProcedure;
import org.simantics.db.procedure.ListenerBase;
import org.simantics.db.request.RequestFlags;
import org.simantics.utils.Development;

/*
 * Size analysis:
 * java 8 byte
 * id 8 byte
 * statusOrException 4 byte
 * p1 = 4 byte
 * p2OrParents = 4 byte
 * result = 4 byte 
 * 
 * total 32byte
 * 
 */

final public class Objects extends CollectionBinaryQuery<IntProcedure> {

	public Objects(final int r1, final int r2) {
		super(r1, r2);
	}

	final static Objects entry(final QueryProcessor provider, final int r1, final int r2) {
		return (Objects)provider.objectsMap.get(r1,r2);
	}

	final static Collection<Objects> entries(final QueryProcessor processor, final int r1) {
		return processor.objectsMap.values(r1);
	}

	public final static void runner(ReadGraphImpl graph, final int r1, final int r2, CacheEntry parent, ListenerBase listener, final IntProcedure procedure) {

		if(parent == null && listener == null) {
			Objects.computeForEach(graph, r1, r2, null, procedure);
			return;
		}

		QueryProcessor processor = graph.processor;

		Objects entry = (Objects)processor.objectsMap.get(r1,r2);
		if(entry == null) {

			entry = new Objects(r1, r2);
			entry.setPending();
			entry.clearResult(processor.querySupport);
			entry.putEntry(processor);

			processor.performForEach(graph, entry, parent, listener, procedure);

		} else {

			if(entry.isPending()) {
				synchronized(entry) {
					if(entry.isPending()) {
						processor.registerDependencies(graph, entry, parent, listener, procedure, false);
						computeForEach(graph, r1, r2, null, procedure);
						return;
					}
				}
			}

			processor.performForEach(graph, entry, parent, listener, procedure);

		}

	}

	static class Runner2Procedure implements IntProcedure {
	    
	    public int single = 0;
	    public Throwable t = null;

	    public void clear() {
	        single = 0;
	        t = null;
	    }
	    
        @Override
        public void execute(ReadGraphImpl graph, int i) {
            if(single == 0) single = i;
            else single = -1;
        }

        @Override
        public void finished(ReadGraphImpl graph) {
            if(single == -1) single = 0;
        }

        @Override
        public void exception(ReadGraphImpl graph, Throwable throwable) {
            single = 0;
            this.t = throwable;
        }
        
        public int get() throws DatabaseException {
            if(t != null) {
                if(t instanceof DatabaseException) throw (DatabaseException)t;
                else throw new DatabaseException(t);
            }
            return single;
        }
	    
	}
	
	static final Runner2Procedure runner2Procedure = new Runner2Procedure();
	
    public final static int runner2(ReadGraphImpl graph, final int r1, final int r2, CacheEntry parent) throws DatabaseException {

        runner2Procedure.clear();

        if(parent == null) {
            Objects.computeForEach(graph, r1, r2, null, runner2Procedure);
            return runner2Procedure.get();
        }

        QueryProcessor processor = graph.processor;

        Objects entry = (Objects)processor.objectsMap.get(r1,r2);
        if(entry == null) {

            entry = new Objects(r1, r2);
            entry.setPending();
            entry.clearResult(processor.querySupport);
            entry.putEntry(processor);

            processor.performForEach(graph, entry, parent, null, runner2Procedure);
            return runner2Procedure.get();

        } else {

            if(entry.isPending()) throw new IllegalStateException();

            processor.performForEach(graph, entry, parent, null, runner2Procedure);
            return runner2Procedure.get();

        }

    }
	
	@Override
	public BinaryQuery<IntProcedure> getEntry(QueryProcessor provider) {
		return provider.objectsMap.get(id);
	}

	@Override
	public void putEntry(QueryProcessor provider) {
		if(Development.DEVELOPMENT) {
			if(Development.<Boolean>getProperty(DevelopmentKeys.QUERYPROCESSOR_PUT, Bindings.BOOLEAN)) {
				System.err.println("put " + this);
			}
		}
		provider.objectsMap.put(id, this);
	}

	@Override
	final public void removeEntry(QueryProcessor provider) {
		provider.objectsMap.remove(id);
	}

	final static private IntArray getAssertionMap(ReadGraphImpl graph, final int r1, final int r2, final Objects entry) {

		class AssertionMapProc implements IntProcedure {

			boolean first = true;

			private IntArray result;

			public void addStatement(int s, int p, int o) {

				if(result.size() == 0) {
					result.add(s);
					result.add(p);
					result.add(o);
				} else {
					for(int i = 0;i < result.sizeOrData ; i+=3) {
						int existingP = result.data[i+1];
						if(p == existingP) {
							int existingO = result.data[i+2];
							if(existingO == o) return;
						}
					}
					result.add(s);
					result.add(p);
					result.add(o);
				}

			}

			@Override
			public void execute(ReadGraphImpl graph, int type) {
				AssertedStatements stms = AssertedStatements.queryEach(graph, type, r2, graph.processor, entry, null, NOPT);
				if(result == null) {
					result = stms.getResult();
				} else {
					if (first) {
						IntArray ia = result;
						result = new IntArray();
						if(ia.data != null) {
							for(int i = 0;i < ia.sizeOrData ; i+=3) addStatement(ia.data[i],ia.data[i+1],ia.data[i+2]);
						}
						first = false;
					}
					IntArray ia = stms.getResult();
					if(ia.data != null) {
						for(int i = 0;i < ia.sizeOrData ; i+=3) addStatement(ia.data[i],ia.data[i+1],ia.data[i+2]);
					}
				}
			}

			@Override
			public void finished(ReadGraphImpl graph) {
			}

			@Override
			public void exception(ReadGraphImpl graph, Throwable throwable) {
			}

		}

		AssertionMapProc amp = new AssertionMapProc();

		// This dependency could be cut
		PrincipalTypes.queryEach(graph, r1, graph.processor, entry, null, amp);

		return amp.result;

	}

	final static private void forSingleAssertion(ReadGraphImpl graph, final int r1, final int r2, final Objects entry, final IntProcedure procedure) {

		IntArray map = getAssertionMap(graph, r1, r2, entry);
		if(map == null) {
			if(entry != null) entry.finish(graph, procedure);
			else procedure.finished(graph);
			return;
		}

		int size = map.size();
		if(size == 3) {

			int value = map.data[2];

			if(entry != null) {
				entry.addOrSetFunctional(value);
				entry.finish(graph, procedure);
			} else {
				procedure.execute(graph, value);
				procedure.finished(graph);
			}

		} else if(size == 0) {

			if(entry != null) entry.finish(graph, procedure);
			else procedure.finished(graph);

		} else {

			int candidateS = map.data[0];
			int candidateO = map.data[2];

			SuperTypes candidate = SuperTypes.queryEach(graph, candidateS, graph.processor, entry, null, NOP);
			if(candidate.isExcepted()) {
				if(entry != null) entry.except((Throwable)candidate.getResult());
				procedure.exception(graph, (Throwable)candidate.getResult());
				return;
			}
			IntSet candidateIs = candidate.getResult();

			for(int i=3;i<map.size();i+=3) {

				int nextS = map.data[i];
				int nextO = map.data[i+2];

				if(nextS != candidateS) {

					if(candidateIs.contains(nextS)) {

						// Next is a super type of candidate => ignore next

					} else {

						SuperTypes next = SuperTypes.queryEach(graph, nextS, graph.processor, entry, null, NOP);
						if(next.isExcepted()) {
							if(entry != null) entry.except((Throwable)next.getResult());
							procedure.exception(graph, (Throwable)next.getResult());
							return;
						}
						IntSet nextIs = next.getResult();

						if(nextIs.contains(candidateS)) {

							// Candidate is a super type of next => next is the new candidate

							candidateS = nextS;
							candidateO = nextO;
							candidateIs = nextIs;

						} else {

							// candidate and next are unrelated => error
							ManyObjectsForFunctionalRelationException exception = new ManyObjectsForFunctionalRelationException("Functional relation has conflicting assertions " + r1 + ", " + r2 + " " + map , r1);

							if(entry != null) entry.except(exception);
							procedure.exception(graph, exception);
							return;	        				

						}

					}

				}

			}

			if(entry != null) {
				entry.addOrSetFunctional(candidateO);
				entry.finish(graph, procedure);
			} else {
				procedure.execute(graph, candidateO);
				procedure.finished(graph);
			}

		}

	}

	final static InternalProcedure<IntSet> NOP = new InternalProcedure<IntSet>() {

		@Override
		public void execute(ReadGraphImpl graph, IntSet result) {
		}

		@Override
		public void exception(ReadGraphImpl graph, Throwable throwable) {
		}

	};

	final static TripleIntProcedure NOPT = new TripleIntProcedure() {


		@Override
		public void exception(ReadGraphImpl graph, Throwable throwable) {
		}

		@Override
		public void execute(ReadGraphImpl graph, int s, int p, int o) {
		}

		@Override
		public void finished(ReadGraphImpl graph) {
		}

	};

	// Search for one statement
	final public void computeFunctionalIndex(ReadGraphImpl graph, final QueryProcessor provider, final RelationInfo ri, final IntProcedure procedure) {
		computeFunctionalIndex(graph, r1(), r2(), this, ri, procedure);
	}

	// Search for one statement
	final static public void computeFunctionalIndex(ReadGraphImpl graph, final int r1, final int r2, final Objects entry, final RelationInfo ri, final IntProcedure procedure) {

		if(ri.isFinal) {

			int result = graph.processor.querySupport.getFunctionalObject(r1, r2);

			if(result == 0) {

				// Check for assertions
				forSingleAssertion(graph, r1, r2, entry, procedure);

			} else if (result == -1) {

				graph.processor.querySupport.getObjects(graph, r1, r2, new IntProcedure() {

					@Override
					public void execute(ReadGraphImpl graph, int i) {
						if(entry != null) entry.addOrSetFunctional(i);
						else procedure.execute(graph, i);
					}

					@Override
					public void exception(ReadGraphImpl graph, Throwable t) {
						if(DebugException.DEBUG) new DebugException(t).printStackTrace();
					}

					@Override
					public void finished(ReadGraphImpl graph) {
					}

				});

				// Check for assertions
				forSingleAssertion(graph, r1, r2, entry, procedure);

			} else {

				// If functional relation was found there is no need to check assertions
				if(entry != null) {
					entry.addOrSetFunctional(result);
					entry.finish(graph, procedure);
				} else {
					procedure.execute(graph, result);
					procedure.finished(graph);
				}

				
			}

		} else {

			// Note! The dependency is intentionally cut!
			DirectPredicates.queryEach(graph, r1, graph.processor, null, null, new SyncIntProcedure() {

				boolean found = false;

				@Override
				public void run(ReadGraphImpl graph) {

					if(found) {
						if(entry != null) entry.finish(graph, procedure);
						else procedure.finished(graph);
					} else {

						// Check for assertions
						forSingleAssertion(graph, r1, r2, entry, procedure);

					}

				}

				@Override
				public void execute(ReadGraphImpl graph, final int pred) {

					if(found) return;

					if(pred == r2) {

						// Note! The dependency is intentionally cut!
						DirectObjects.queryEach(graph, r1, pred, graph.processor, null, null, new IntProcedure() {

							@Override
							public void execute(ReadGraphImpl graph, int i) {

								if(!found) {

									if(entry != null) entry.addOrSetFunctional(i);
									else procedure.execute(graph, i);

									found = true;

								} else {

									ManyObjectsForFunctionalRelationException exception = new ManyObjectsForFunctionalRelationException("Functional relation has more than one statement (r1=" + r1 + ", r2=" + r2 + ").", r1);
									if(entry != null) entry.except(exception);
									procedure.exception(graph, exception);

								}

							}

							@Override
							public void finished(ReadGraphImpl graph) {
							}

							@Override
							public void exception(ReadGraphImpl graph, Throwable t) {
								procedure.exception(graph, t);
							}

						});

					} else {

						SuperRelations.queryEach(graph, pred, graph.processor, entry, null, new InternalProcedure<IntSet>() {

							@Override
							public void execute(ReadGraphImpl graph, IntSet result) {

								if(found) return;

								if(result.contains(r2)) {

									// Note! The dependency is intentionally cut!
									DirectObjects.queryEach(graph, r1, pred, graph.processor, null, null, new IntProcedure() {

										@Override
										public void execute(ReadGraphImpl graph, int i) {

											if(!found) {

												if(entry != null) entry.addOrSetFunctional(i);
												else procedure.execute(graph, i);

												found = true;

											} else {

												ManyObjectsForFunctionalRelationException exception = new ManyObjectsForFunctionalRelationException("Functional relation has more than one statement (r1=" + r1 + ", r2=" + r2 + ").", r1);
												if(entry != null) entry.except(exception);
												procedure.exception(graph, exception);

											}

										}

										@Override
										public void finished(ReadGraphImpl graph) {
										}

										@Override
										public void exception(ReadGraphImpl graph, Throwable t) {
											procedure.exception(graph, t);
										}

									});

								}

							}

							@Override
							public void exception(ReadGraphImpl graph, Throwable t) {
								procedure.exception(graph, t);
							}

						});

					}

				}

				@Override
				public void finished(ReadGraphImpl graph) {

					dec(graph);

				}

			});


		}

	}

	final static private void forAssertions(ReadGraphImpl graph, final int r1, final int r2, final Objects entry, final IntProcedure procedure) {

		// Note! The dependency is intentionally cut!
		PrincipalTypes.queryEach(graph, r1, graph.processor, null, null, new SyncIntProcedure() {

			@Override
			public void run(ReadGraphImpl graph) {

				if(entry != null) entry.finish(graph, procedure);
				else procedure.finished(graph);

			}

			TripleIntProcedure proc = new TripleIntProcedure() {

				@Override
				public void execute(ReadGraphImpl graph, int s, int p, int o) {
					if(entry != null) entry.addOrSet(o);
					else procedure.execute(graph, o);
				}

				@Override
				public void finished(ReadGraphImpl graph) {
					dec(graph);
				}

				@Override
				public void exception(ReadGraphImpl graph, Throwable t) {
					if(DebugException.DEBUG) new DebugException(t).printStackTrace();
					procedure.exception(graph, t);
					dec(graph);
				}

			};

			@Override
			public void execute(ReadGraphImpl graph, int type) {

				inc();

				AssertedStatements.queryEach(graph, type, r2, graph.processor, entry, null, proc);

			}

			@Override
			public void finished(ReadGraphImpl graph) {
				dec(graph);
			}

		});


	}

	final public static void computeNotFunctionalFinalIndex(ReadGraphImpl graph, final int r1, final int r2, final QueryProcessor provider, RelationInfo ri, AsyncMultiProcedure<Resource> procedure) {

		throw new Error();

	}

	final public void computeNotFunctionalIndex(ReadGraphImpl graph, RelationInfo ri, final IntProcedure procedure) {
		computeNotFunctionalIndex(graph, r1(), r2(), this, ri, procedure);
	}

	final static public void computeNotFunctionalIndex(ReadGraphImpl graph, final int r1, final int r2, final Objects entry, RelationInfo ri, final IntProcedure procedure) {

		if(ri.isFinal) {

			graph.processor.querySupport.getObjects(graph, r1, r2, new IntProcedure() {

				@Override
				public void execute(ReadGraphImpl graph, int i) {
					if(entry != null) entry.addOrSet(i);
					else procedure.execute(graph, i);
				}

				@Override
				public void exception(ReadGraphImpl graph, Throwable t) {
					if(DebugException.DEBUG) new DebugException(t).printStackTrace();
					procedure.exception(graph, t);
				}

				@Override
				public void finished(ReadGraphImpl graph) {
				}

			});

			if(ri.isAsserted) {
				forAssertions(graph, r1, r2, entry, procedure);
			} else {
				if(entry != null) entry.finish(graph, procedure);
				else procedure.finished(graph);
			}

		} else {

			// Note! The dependency is intentionally cut!
			DirectPredicates.queryEach(graph, r1, graph.processor, null, null, new SyncIntProcedure() {

				@Override
				public void run(ReadGraphImpl graph) {

					forAssertions(graph, r1, r2, entry, procedure);

				}

				@Override
				public void execute(ReadGraphImpl graph, final int pred) {

					if(pred == r2) {

						inc();

						// Note! The dependency is intentionally cut!
						DirectObjects.queryEach(graph, r1, pred, graph.processor, null, null, new IntProcedure() {

							@Override
							public void execute(ReadGraphImpl graph, int i) {
								if(entry != null) entry.addOrSet(i);
								else procedure.execute(graph, i);
							}

							@Override
							public void finished(ReadGraphImpl graph) {
								dec(graph);
							}

							@Override
							public void exception(ReadGraphImpl graph, Throwable t) {
								procedure.exception(graph, t);
								dec(graph);
							}

						});

					} else {

						inc();

						SuperRelations.queryEach(graph, pred, graph.processor, entry, null, new InternalProcedure<IntSet>() {

							@Override
							public void execute(ReadGraphImpl graph, IntSet result) {

								if(result.contains(r2)) {

									inc();

									// Note! The dependency is intentionally cut!
									DirectObjects.queryEach(graph, r1, pred, graph.processor, null, null, new IntProcedure() {

										@Override
										public void execute(ReadGraphImpl graph, int i) {
											if(entry != null) entry.addOrSet(i);
											else procedure.execute(graph, i);
										}

										@Override
										public void finished(ReadGraphImpl graph) {
											dec(graph);
										}

										@Override
										public void exception(ReadGraphImpl graph, Throwable t) {
											if(DebugException.DEBUG) new DebugException(t).printStackTrace();
											procedure.exception(graph, t);
											dec(graph);
										}

									});

								}

								dec(graph);

							}

							@Override
							public void exception(ReadGraphImpl graph, Throwable t) {
								procedure.exception(graph, t);
								dec(graph);
							}

						});

					}

				}

				@Override
				public void finished(ReadGraphImpl graph) {
					dec(graph);
				}

			});

		}

	}

	@Override
	public void computeForEach(ReadGraphImpl graph, final QueryProcessor provider, final IntProcedure procedure, final boolean store) {
		computeForEach(graph, r1(), r2(), this, procedure);
	}

	public static void computeForEach(ReadGraphImpl graph, final int r1, final int r2, final Objects entry, final IntProcedure procedure) {

		RelationInfo ri = RelationInfoQuery.queryEach(graph, r2, graph.processor, entry, null, ip);
		graph.ensureLoaded(r1, r2);       
		if(ri.isFunctional) {
			computeFunctionalIndex(graph, r1, r2, entry, ri, procedure);
		} else {
			computeNotFunctionalIndex(graph, r1, r2, entry, ri, procedure);
		}

	}

	final static InternalProcedure<RelationInfo> ip = new InternalProcedure<RelationInfo>() {

		@Override
		public void execute(ReadGraphImpl graph, RelationInfo result) {
		}

		@Override
		public void exception(ReadGraphImpl graph, Throwable throwable) {
		}

	};

	@Override
	public String toString() {
		return "Objects[" + r1() + " - " + r2() + "]";
	}

	final private void finish(ReadGraphImpl graph, IntProcedure procedure) {

		assert(assertPending());

		synchronized(this) {
			setReady();
		}

		IntArray v = (IntArray)getResult();

		if(v.data == null) {
			if(v.sizeOrData != IntArray.NO_DATA) {
				procedure.execute(graph, v.sizeOrData);
			}
		} else {
			for(int i = 0;i < v.sizeOrData ; i++) {
				procedure.execute(graph, v.data[i]);
			}
		}

		procedure.finished(graph);

	}

	final public void addOrSet(int add) {

		assert(assertPending());

		IntArray value = (IntArray)getResult();
		synchronized(value) {
			value.add(add);
		}

	}

	final public void addOrSetFunctional(int add) {

		assert(isPending());

		IntArray value = (IntArray)getResult();
		value.add(add);

	}

	@Override
	public void performFromCache(ReadGraphImpl graph, QueryProcessor provider, final IntProcedure procedure) {

		assert(isReady());

		if(handleException(graph, procedure)) return;

		final IntArray value = (IntArray)getResult();
		if(value.data == null) {
			if(value.sizeOrData != IntArray.NO_DATA) procedure.execute(graph, value.sizeOrData);
		} else {
			for(int i = 0;i < value.sizeOrData ; i++) procedure.execute(graph, value.data[i]);
		}

		procedure.finished(graph);

	}

	@Override
	public void recompute(ReadGraphImpl graph, QueryProcessor provider) {

		final Semaphore s = new Semaphore(0);

		computeForEach(graph, provider, new IntProcedureAdapter() {

			@Override
			public void finished(ReadGraphImpl graph) {
				s.release();
			}

			@Override
			public void exception(ReadGraphImpl graph, Throwable t) {
				s.release();
				new Error("Error in recompute.", t).printStackTrace();
			}

		}, true);

		while(!s.tryAcquire()) {
			provider.resume(graph);
		}

	}

	@Override
	public int type() {
		return RequestFlags.IMMEDIATE_UPDATE;
	}

	@Override
	boolean isImmutable(ReadGraphImpl graph) {
		return graph.processor.isImmutable(r1());
	}

}
