/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import java.util.concurrent.Semaphore;

import org.simantics.db.common.exception.DebugException;
import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.impl.procedure.IntProcedureAdapter;
import org.simantics.db.procedure.ListenerBase;

final public class DirectPredicates extends CollectionUnaryQuery<IntProcedure> {

	private DirectPredicates(final int resource) {
		super(resource);
	}

	final static DirectPredicates entry(final QueryProcessor provider, final int r) {

		return (DirectPredicates)provider.directPredicatesMap.get(r);

	}

	final static void runner(ReadGraphImpl graph, final int r, final QueryProcessor provider, final CacheEntry parent, final ListenerBase listener, final IntProcedure procedure) {

		DirectPredicates entry = (DirectPredicates)provider.directPredicatesMap.get(r);
		if(entry == null) {

			entry = new DirectPredicates(r);
        	entry.setPending();
        	entry.clearResult(provider.querySupport);
        	entry.putEntry(provider);
			
			provider.performForEach(graph, entry, parent, listener, procedure);
			
		} else {
			
			if(!entry.isReady()) {
				synchronized(entry) {
					if(!entry.isReady()) {
						provider.registerDependencies(graph, entry, parent, listener, procedure, false);
						entry.computeForEach(graph, provider, procedure, true);
						return;
					}
				}
			}
			
			provider.performForEach(graph, entry, parent, listener, procedure);
			
		}

	}

	final public static void queryEach(ReadGraphImpl graph, final int r, final QueryProcessor provider, final CacheEntry parent, final ListenerBase listener, final IntProcedure procedure) {

		if(parent == null && listener == null) {
			DirectPredicates entry = (DirectPredicates)provider.directPredicatesMap.get(r);
			if(entry != null && entry.isReady()) { 
				entry.performFromCache(graph, provider, procedure);
				return;
			} else {
			    computeForEach(graph, r, null, procedure, false);
			    return;
			}
		}

		runner(graph, r, provider, parent, listener, procedure);

	}

	
	@Override
	public void clearResult(QuerySupport support) {
		// The cached result is never used
		setResult(INVALID_RESULT);
	}
	
	@Override
	public UnaryQuery<IntProcedure> getEntry(QueryProcessor provider) {
		return provider.directPredicatesMap.get(id);
	}

	@Override
	public void putEntry(QueryProcessor provider) {
		provider.directPredicatesMap.put(id, this);
	}

	@Override
	final public void removeEntry(QueryProcessor provider) {
		provider.directPredicatesMap.remove(id);
	}

	@Override
	public Object computeForEach(ReadGraphImpl graph, QueryProcessor provider, IntProcedure procedure, boolean store) {
	    return computeForEach(graph, id, this, procedure, store);
	}
	
	public static Object computeForEach(ReadGraphImpl graph, int id, final DirectPredicates entry, final IntProcedure procedure, final boolean store) {

		graph.processor.querySupport.ensureLoaded(graph, id);
		
		final IntArray list = new IntArray();

		graph.processor.querySupport.getPredicates(graph, id, new IntProcedure() {

			@Override
			public void execute(ReadGraphImpl graph, int i) {
				list.add(i);
			}

			@Override
			public void finished(ReadGraphImpl graph) {
			}

			@Override
			public void exception(ReadGraphImpl graph, Throwable t) {
				if(DebugException.DEBUG) new DebugException(t).printStackTrace();
			}

		});

		if(entry != null)
		    entry.finish(graph, graph.processor);

		if(list.data == null) {
			if(list.sizeOrData != IntArray.NO_DATA) procedure.execute(graph, list.sizeOrData);
		} else {
			for(int i = 0;i < list.sizeOrData ; i++) procedure.execute(graph, list.data[i]);
		}

		procedure.finished(graph);
		
		return list;

	}

	@Override
	public String toString() {
		return "DirectPredicates[" + id + "]";
	}

	@Override
	public void setReady() {
		statusOrException = READY;
	}
	
	final private void finish(ReadGraphImpl graph, QueryProcessor provider) {

		setReady();

	}

	@Override
	public Object performFromCache(ReadGraphImpl graph, QueryProcessor provider, IntProcedure procedure) {

		assert(isReady());

		return computeForEach(graph, provider, procedure, false);

	}

	@Override
	public void recompute(ReadGraphImpl graph, QueryProcessor provider) {

		final Semaphore s = new Semaphore(0);

		computeForEach(graph, provider, new IntProcedureAdapter() {

			@Override
			public void finished(ReadGraphImpl graph) {
				s.release();
			}

			@Override
			public void exception(ReadGraphImpl graph, Throwable t) {
				s.release();
				new Error("Error in recompute.", t).printStackTrace();
			}

		}, true);

        while(!s.tryAcquire()) {
        	provider.resume(graph);
        }

	}


    @Override
    boolean isImmutable(ReadGraphImpl graph) {
    	return graph.processor.isImmutable(id);
    }

}
