package org.simantics.spreadsheet.graph.synchronization;

import java.util.Collection;
import java.util.Map;

import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.datatypes.literal.Font;
import org.simantics.datatypes.literal.RGB;
import org.simantics.spreadsheet.graph.SpreadsheetBook;
import org.simantics.spreadsheet.graph.SpreadsheetStyle;
import org.simantics.structural.synchronization.base.CommandBuilder;
import org.simantics.structural.synchronization.base.ModuleUpdateContext;
import org.simantics.structural.synchronization.base.ModuleUpdaterBase;
import org.simantics.structural.synchronization.base.PropertyUpdateRule;
import org.simantics.structural.synchronization.base.Solver;

public class StyleUpdater extends ModuleUpdaterBase<SheetLineComponent> {

    private static class ForegroundUpdateRule implements PropertyUpdateRule<SheetLineComponent> {
        @Override
        public String getPropertyName() {
            return "foreground";
        }
        
        @Override
        public void apply(ModuleUpdateContext<SheetLineComponent> context, boolean isCreating, Map<String, Variant> propertyMap, Map<String, Collection<String>> connectionMap, Variant value) {
            StyleCommandBuilder builder = context.getConcreteCommand();
            RGB.Integer color = (RGB.Integer) value.getValue();
            builder.foreground = color;
        }
    }
    
    private static class BackgroundUpdateRule implements PropertyUpdateRule<SheetLineComponent> {
        @Override
        public String getPropertyName() {
            return "background";
        }

        @Override
        public void apply(ModuleUpdateContext<SheetLineComponent> context, boolean isCreating, Map<String, Variant> propertyMap, Map<String, Collection<String>> connectionMap, Variant value) {
            StyleCommandBuilder builder = context.getConcreteCommand();
            RGB.Integer color = (RGB.Integer) value.getValue();
            builder.background = color;
        }
    }
    
    private static class FontUpdateRule implements PropertyUpdateRule<SheetLineComponent> {
        @Override
        public String getPropertyName() {
            return "font";
        }

        @Override
        public void apply(ModuleUpdateContext<SheetLineComponent> context, boolean isCreating, Map<String, Variant> propertyMap, Map<String, Collection<String>> connectionMap, Variant value) {
            StyleCommandBuilder builder = context.getConcreteCommand();
            Font font = (Font) value.getValue();
            builder.font = font;
        }
    }

    private static class AlignUpdateRule implements PropertyUpdateRule<SheetLineComponent> {
        @Override
        public String getPropertyName() {
            return "align";
        }

        @Override
        public void apply(ModuleUpdateContext<SheetLineComponent> context, boolean isCreating, Map<String, Variant> propertyMap, Map<String, Collection<String>> connectionMap, Variant value) {
            StyleCommandBuilder builder = context.getConcreteCommand();
            int align = (int) value.getValue();
            builder.align = align;
        }
    }
    
    private static class FormatStringUpdateRule implements PropertyUpdateRule<SheetLineComponent> {
        @Override
        public String getPropertyName() {
            return "formatString";
        }

        @Override
        public void apply(ModuleUpdateContext<SheetLineComponent> context, boolean isCreating, Map<String, Variant> propertyMap, Map<String, Collection<String>> connectionMap, Variant value) {
            StyleCommandBuilder builder = context.getConcreteCommand();
            String formatString = (String) value.getValue();
            builder.formatString = formatString;
        }
    }

    private static class FormatIndexUpdateRule implements PropertyUpdateRule<SheetLineComponent> {
        @Override
        public String getPropertyName() {
            return "formatIndex";
        }

        @Override
        public void apply(ModuleUpdateContext<SheetLineComponent> context, boolean isCreating, Map<String, Variant> propertyMap, Map<String, Collection<String>> connectionMap, Variant value) {
            StyleCommandBuilder builder = context.getConcreteCommand();
            int index = (int) value.getValue();
            builder.formatIndex = index;
        }
    }

    private static class BorderUpdateRule implements PropertyUpdateRule<SheetLineComponent> {
        @Override
        public String getPropertyName() {
            return "border";
        }

        @Override
        public void apply(ModuleUpdateContext<SheetLineComponent> context, boolean isCreating, Map<String, Variant> propertyMap, Map<String, Collection<String>> connectionMap, Variant value) {
            StyleCommandBuilder builder = context.getConcreteCommand();
            int border = (int) value.getValue();
            builder.border = border;
        }
    }
    
    private static class LockedUpdateRule implements PropertyUpdateRule<SheetLineComponent> {
        @Override
        public String getPropertyName() {
            return "locked";
        }

        @Override
        public void apply(ModuleUpdateContext<SheetLineComponent> context, boolean isCreating, Map<String, Variant> propertyMap, Map<String, Collection<String>> connectionMap, Variant value) {
            StyleCommandBuilder builder = context.getConcreteCommand();
            boolean locked = (boolean) value.getValue();
            builder.locked = locked;
        }
    }
    
    private static class ColumnSpanUpdateRule implements PropertyUpdateRule<SheetLineComponent> {
        @Override
        public String getPropertyName() {
            return "columnSpan";
        }

        @Override
        public void apply(ModuleUpdateContext<SheetLineComponent> context, boolean isCreating, Map<String, Variant> propertyMap, Map<String, Collection<String>> connectionMap, Variant value) {
            StyleCommandBuilder builder = context.getConcreteCommand();
            int columnSpan = (int) value.getValue();
            builder.columnSpan = columnSpan;
        }
    }
    
    private static class RowSpanUpdateRule implements PropertyUpdateRule<SheetLineComponent> {
        @Override
        public String getPropertyName() {
            return "rowSpan";
        }

        @Override
        public void apply(ModuleUpdateContext<SheetLineComponent> context, boolean isCreating, Map<String, Variant> propertyMap, Map<String, Collection<String>> connectionMap, Variant value) {
            StyleCommandBuilder builder = context.getConcreteCommand();
            int rowSpan = (int) value.getValue();
            builder.rowSpan = rowSpan;
        }
    }

    
    public StyleUpdater(String moduleType) {
        super(moduleType);
        
        addPropertyUpdateRule(new ForegroundUpdateRule());
        addPropertyUpdateRule(new BackgroundUpdateRule());
        addPropertyUpdateRule(new FontUpdateRule());
        addPropertyUpdateRule(new AlignUpdateRule());
        addPropertyUpdateRule(new FormatStringUpdateRule());
        addPropertyUpdateRule(new FormatIndexUpdateRule());
        addPropertyUpdateRule(new BorderUpdateRule());
        addPropertyUpdateRule(new LockedUpdateRule());
        addPropertyUpdateRule(new ColumnSpanUpdateRule());
        addPropertyUpdateRule(new RowSpanUpdateRule());
    }

    @Override
    public CommandBuilder createAddCommandBuilder(final String name) {
        return new StyleCommandBuilder(name, false);
    }

    @Override
    public CommandBuilder createUpdateCommandBuilder(String name) {
        return new StyleCommandBuilder(name, true);
    }
    
    private static class StyleCommandBuilder implements CommandBuilder {

        protected int rowSpan;
        protected int columnSpan;
        protected int border;
        protected boolean locked;
        protected int formatIndex;
        protected String formatString;
        protected org.simantics.datatypes.literal.RGB.Integer background;
        protected org.simantics.datatypes.literal.RGB.Integer foreground;
        protected Font font;
        protected int align;
        
        private boolean update;
        private String name;
        
        public StyleCommandBuilder(String name, boolean update) {
            this.name = name.split("/")[1];
            this.update = update;
        }
        
        @Override
        public void apply(Solver solver) {
            SpreadsheetBook book = solver.getConcreteSolver();
            SpreadsheetStyle style = SpreadsheetStyle.newInstace().name(name).align(align).font(font)
                    .background(background).foreground(foreground).border(border).locked(locked).rowSpan(rowSpan)
                    .columnSpan(columnSpan).formatIndex((short) formatIndex).formatString(formatString).build();
            if (book.getStyle(style.getStyleId()) == null)
                book.addStyle(style);
            
        }

        @SuppressWarnings("unchecked")
        @Override
        public <T> T getConcrete() {
            return (T) this;
        }
        
        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append("StyleCommandBuilder for ").append(name).append(" [background=").append(background).append(", foregroung=").append(foreground).append(", font=").append(font).append(", align=").append(align).append("]");
            return sb.toString();
        }
        
    }
    
}
