package org.simantics.graph.refactoring;

import java.io.BufferedInputStream;
import java.io.DataInput;
import java.io.DataInputStream;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;

import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.databoard.container.DataContainer;
import org.simantics.databoard.container.DataContainers;
import org.simantics.graph.representation.PrettyPrintTG;
import org.simantics.graph.representation.TransferableGraph1;

/**
 * @author Antti Villberg
 * @since 1.24.0
 */
public class FixExportedOntology {
	
	static TransferableGraph1 convertExportedSharedOntologyIntoBundleOntology(Path input, Path output) throws Exception {
		System.out.format("Converting exported shared ontology%n\t" + input.toString() + "%nto bundle-compatible ontology%n\t" + output.toString());
		try (InputStream is = new BufferedInputStream(Files.newInputStream(input), 128*1024)) {
			DataInput dis = new DataInputStream(is);
			org.simantics.databoard.container.DataContainer container = 
					DataContainers.readFile(dis); 
			Binding binding = TransferableGraph1.BINDING;
			TransferableGraph1 graph = (TransferableGraph1)container.content.getValue(binding);
			GraphRefactoringUtils.fixOntologyExport(graph);

			DataContainers.writeFile(output.toFile(), new DataContainer(
					container.format, container.version,
					container.metadata, new Variant(TransferableGraph1.BINDING, graph)));
			
			return graph;
			
		}
	}
	
	public static void createTg(Path input) throws Exception {
	    Path output = input.getParent().resolve("graph.tg");
	    createTg(input, output);
	}
	
	private static void createTg(Path input, Path output) throws Exception {
	    convertExportedSharedOntologyIntoBundleOntology(input, output);
	}
	
	public static void createTGAndPGraph(Path input) throws Exception {
        Path output1 = input.getParent().resolve("graph.tg");
        TransferableGraph1 tg = convertExportedSharedOntologyIntoBundleOntology(input, output1);
        String listing = PrettyPrintTG.print(tg, false);
        String newName = input.getFileName().toString();
        if (newName.contains("."))
            newName = newName.split("\\.")[0];
        Path output2 = input.getParent().resolve(newName + ".pgraph");
        Files.write(output2, listing.getBytes(),StandardOpenOption.CREATE);
	}

	public static void main(String[] args) throws Exception {
		if (args.length == 0) {
			System.out.println("Required arguments: <input .sharedLibrary file> [<output .tg file>]");
		} else if (args.length == 1) {
			Path input = Paths.get(args[0]);
			createTGAndPGraph(input);
		} else {
			convertExportedSharedOntologyIntoBundleOntology(Paths.get(args[0]), Paths.get(args[1]));
		}
	}

}
