/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.procore.cluster;

import java.io.PrintStream;

import org.simantics.db.Session;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.ClusterBase;
import org.simantics.db.impl.ClusterI;
import org.simantics.db.impl.ClusterI.CompleteTypeEnum;
import org.simantics.db.impl.ClusterTranslator;
import org.simantics.db.service.ClusterUID;

import fi.vtt.simantics.procore.internal.SessionImplSocket;
import gnu.trove.map.hash.TIntIntHashMap;

final public class TestCluster {
    private static final boolean DEBUG = false;
    private static final int AddedResources = 100; //ClusterTraits.getMaxNumberOfResources();
    private static final int AddedStatements = 100; // Must be less than or equal to number of resources.
    private static final int ValueCapacity = 5;
    private final ClusterBase builtinCluster;
    private final ClusterUID BuiltinClusterUID = ClusterUID.Builtin;
    private final int BuiltinClusterKey;
    private final ClusterBase foreignCluster;
    private final ClusterUID ForeignClusterUID = ClusterUID.make(/*ClusterUID.Builtin.first*/0, ClusterUID.Builtin.second+1);
    private final ClusterBase objectCluster;
    private final ClusterUID ObjectClusterUID = ClusterUID.make(/*ClusterUID.Builtin.first*/0, ClusterUID.Builtin.second+2);
    private final ClusterTranslator support;
    PrintStream out = System.out;
    TestCluster(Session session) {
        support = ((SessionImplSocket)session).clusterTranslator;
        builtinCluster = support.getClusterByClusterUIDOrMake(BuiltinClusterUID);
        BuiltinClusterKey = builtinCluster.getClusterKey();
        foreignCluster = support.getClusterByClusterUIDOrMake(ForeignClusterUID);
        objectCluster = support.getClusterByClusterUIDOrMake(ObjectClusterUID);
    }
    private void testReading(ClusterI cluster, int[] resourceKeys, int[] statementKeys, int[] objectKeys, boolean onePredicateOn, int AddedStatements, boolean foreignClusterOn)
    throws DatabaseException {
        Stopwatch sw = new Stopwatch();
        sw.start();
        for (int i = 0; i < AddedResources; ++i) {
            int resourceKey = resourceKeys[i];
            CompleteTypeEnum ct = cluster.getCompleteType(resourceKey, support);
            if (ct != CompleteTypeEnum.NotComplete) {
                int ck = cluster.getCompleteObjectKey(resourceKey, support);
                Assert(0 != ck);
                Assert(objectKeys[0] == ck);
            }
            final TIntIntHashMap predicates = new TIntIntHashMap();
            final TIntIntHashMap countMap = new TIntIntHashMap();
            countMap.clear();
            ClusterI.ObjectProcedure<TIntIntHashMap> readObjects = new ClusterI.ObjectProcedure<TIntIntHashMap>() {
                @Override
                public boolean execute(final TIntIntHashMap objects, final int object) {
                    int i = objects.get(object);
                    objects.put(object, ++i);
                    return false; // continue looping
                }
            };
            ClusterI.PredicateProcedure<TIntIntHashMap> readPredicates = new ClusterI.PredicateProcedure<TIntIntHashMap>() {
                @Override
                public boolean execute(TIntIntHashMap set, int predicateKey, int objectIndex) {
                    set.put(predicateKey, objectIndex);
                    int i = countMap.get(predicateKey);
                    countMap.put(predicateKey, ++i);
                    return false; // continue looping
                }
            };
            cluster.forPredicates(resourceKey, readPredicates, predicates, support);
            if (onePredicateOn)
                Assert(predicates.size() == 1);
            else {
                Assert(predicates.size() == AddedStatements);
                //Assert(cluster.isComplete(fyi, resourceKey, support));
                //Assert(cluster.getCompleteType(fyi, resourceKey, support) == ClusterI.CompleteTypeEnum.InstanceOf);
            }
            for (int j = 0; j < AddedStatements; ++j) {
                int pKey = onePredicateOn ? resourceKey : statementKeys[j];
                if (onePredicateOn && foreignClusterOn)
                    pKey = statementKeys[0];
                Assert(predicates.contains(pKey));
                Assert(countMap.get(pKey) == 1);
                int oIndex = predicates.get(pKey);
                if (AddedStatements < 3)
                    Assert(oIndex == 0);
                TIntIntHashMap objects = new TIntIntHashMap();
                TIntIntHashMap objects2 = new TIntIntHashMap();
                cluster.forObjects(resourceKey, pKey, oIndex, readObjects, objects, support);
                cluster.forObjects(resourceKey, pKey, readObjects, objects2, support);
                Assert(objects.size() == objects2.size());
                if (onePredicateOn) {
                    Assert(objects.size() == AddedStatements);
                    for (int k = 0; k < AddedStatements; ++k) {
                        int oKey = objectKeys[k];
                        Assert(objects.contains(oKey));
                        Assert(1 == objects.get(oKey));
                        Assert(objects2.contains(oKey));
                        Assert(1 == objects2.get(oKey));
                    }
                } else {
                    Assert(objects.size() == 1);
                    int oKey = objectKeys[j];
                    Assert(objects.contains(oKey));
                    Assert(1 == objects.get(oKey));
                    Assert(objects2.contains(oKey));
                    Assert(1 == objects2.get(oKey));
                }
            }
            if (!cluster.hasValue(resourceKey, support))
                throw new RuntimeException("hasValue() failed for resource key=" + resourceKey);
            byte[] data = cluster.getValue(resourceKey, support);
            Assert(data.length == ValueCapacity);
            for (int l = 0; l < ValueCapacity; ++l)
                Assert((byte) l == data[l]);
        }
        sw.stop();
        out.println("Elapsed time in millseconds " + sw.elapsedMilli() + " for reading.");
        // cluster.check();
        // cluster.printDebugInfo(-1, "koss: ", support);
        out.println("Used space consumption in bytes: " + cluster.getUsedSpace());

    }
    private int getResourceKey(int clusterKey, int resourceIndex)
    throws DatabaseException {
        return ClusterTraits.createResourceKey(clusterKey, resourceIndex);
    }
    private boolean clusterTest(boolean onePredicateOn, boolean foreignClusterOn, int AddedStatements)
    throws DatabaseException {
        String testName = "Cluster ";
        out.println("********************************************");
        String eo = onePredicateOn ? "on" : "off";
        String fco = foreignClusterOn ? "on" : "off";
        out.println(testName + " test with one predicate " + eo + ", foreign cluster " + fco);

        Stopwatch sw = new Stopwatch();
        sw.start();

        ClusterI cluster = ClusterImpl.make(BuiltinClusterUID, BuiltinClusterKey, support);
        if (DEBUG)
            System.out.println("cluster key=" + cluster.getClusterKey() + " id=" + cluster.getClusterId() + " uid=" + cluster.getClusterUID());
        byte[] value = new byte[ValueCapacity];
        for (int i = 0; i < ValueCapacity; ++i)
            value[i] = (byte) i;

        Assert(AddedResources <= ClusterTraits.getMaxNumberOfResources());
        int[] resourceKeys = new int[AddedResources];
        for (int i = 0; i < AddedResources; ++i) {
            resourceKeys[i] = cluster.createResource(support);
            Assert(ClusterTraits.getResourceIndexFromResourceKey(resourceKeys[i]) == i + 1);
            Assert(!cluster.isComplete(resourceKeys[i], support));
        }
        int[] statementKeys = new int[AddedStatements];
        int[] objectKeys = new int[AddedStatements];
        for (int i = 0; i < AddedStatements; ++i) {
            if (foreignClusterOn) {
                Assert(AddedStatements <= ClusterTraits.getMaxNumberOfResources());
                statementKeys[i] = getResourceKey(foreignCluster.getClusterKey(), i + 1);
            } else {
                Assert(AddedStatements <= AddedResources);
                statementKeys[i] = getResourceKey(builtinCluster.getClusterKey(), i + 1);
            }
            objectKeys[i] = getResourceKey(objectCluster.getClusterKey(), i + 1);
        }
        for (int i = 0; i < AddedResources; ++i) {
            final int resourceKey = resourceKeys[i];
            for (int j = 0; j < AddedStatements; ++j) {
                int sKey = resourceKey;
                int pKey = onePredicateOn ? resourceKey : statementKeys[j];
                if (onePredicateOn && foreignClusterOn)
                    pKey = statementKeys[0];
                int oKey = objectKeys[j];
                cluster = cluster.addRelation(sKey, pKey, oKey, support);
                if (null == cluster)
                    throw new RuntimeException("AddRelation() failed.");
                if (null != cluster.addRelation(sKey, pKey, oKey, support))
                    throw new RuntimeException("AddRelation() failed.");
            }
            cluster = cluster.setValue(resourceKey, value, value.length, support);
            if (!cluster.hasValue(resourceKey, support))
                throw new RuntimeException("AddRelation() failed.");
            byte[] data = cluster.getValue(resourceKey, support);
            Assert(data.length == value.length);
            for (int l = 0; l < value.length; ++l)
                Assert((byte) l == data[l]);
        }

        sw.stop();
        out.println("Elapsed time in milliseconds " + sw.elapsedMilli() + " for adding " + AddedResources + " Resources with " + AddedStatements + " statements.");

        testReading(cluster, resourceKeys, statementKeys, objectKeys, onePredicateOn, AddedStatements, foreignClusterOn);

        if (!(cluster instanceof ClusterBig)) {
            sw.restart();
            ClusterI big = ((ClusterImpl)cluster).toBig(support);
            sw.stop();
            out.println("Elapsed time in milliseconds " + sw.elapsedMilli() + " for converting to big.");
            testReading(big, resourceKeys, statementKeys, objectKeys, onePredicateOn, AddedStatements, foreignClusterOn);
        }

        sw.restart();
        for (int i = 0; i < AddedResources; ++i) {
            int resourceKey = resourceKeys[i];
            for (int j = 0; j < AddedStatements; ++j) {
                int sKey = resourceKey;
                int pKey = onePredicateOn ? resourceKey : statementKeys[j];
                if (onePredicateOn && foreignClusterOn)
                    pKey = statementKeys[0];
                int oKey = objectKeys[j];
                if (!cluster.removeRelation(sKey, pKey, oKey, support))
                    throw new RuntimeException("RemoveRelation() failed.");
                if (cluster.removeRelation(sKey, pKey, oKey, support))
                    throw new RuntimeException("RemoveRelation() failed.");
                cluster.denyRelation(sKey, pKey, oKey, support);
            }
            if (!cluster.removeValue(resourceKey, support))
                throw new RuntimeException("removeValue() failed.");
            if (cluster.removeValue(resourceKey, support))
                throw new RuntimeException("removeValue() failed.");
        }
        sw.stop();
        out.println("Elapsed time in millseconds " + sw.elapsedMilli() + " for deleting.");
        out.println("Used space consumption in bytes: " + cluster.getUsedSpace());
        return false; // ok
    }

    private void testBasic()
    throws DatabaseException {
        final int N = AddedStatements;
        support.setStreamOff(true);
        try {
            for (int i = N; i < N + 1; ++i) {
                clusterTest(false, false, i);
                clusterTest(false, true, i);
                clusterTest(true, false, i);
                clusterTest(true, true, i);
            }
        } finally {
            support.setStreamOff(false);
        }
    }

    private void Assert(boolean condition)
    throws DatabaseException {
        if (condition)
            return;
        DatabaseException e = new DatabaseException("Test failed!");
        e.printStackTrace(out);
        throw e;
    }
    public static void test(Session session) {
        TestCluster tc = new TestCluster(session);
        try {
            System.out.println("Begin of tests.");
            tc.testBasic();
            System.out.println("End of tests.");
        } catch (Throwable t) {
            throw new RuntimeException("Tests failed.", t);
        }
    }
    public static void main(String[] args) {
        test(null);
    }
}
