/*******************************************************************************
 * Copyright (c) 2016 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.db.layer0.util;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.simantics.datatypes.literal.GUID;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Statement;
import org.simantics.db.common.request.PossibleIndexRoot;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.impl.EntityInstances.QueryIndex;
import org.simantics.db.layer0.util.ConsistsOfProcess.ConsistsOfProcessEntry;
import org.simantics.db.layer0.util.DomainProcessor3.ExclusionDecision;
import org.simantics.db.layer0.util.TransferableGraphConfiguration2.SeedSpec;
import org.simantics.layer0.Layer0;
import org.simantics.scl.runtime.function.Function1;
import org.simantics.utils.datastructures.Pair;

/**
 * @author Antti Villberg
 * @author Tuukka Lehtonen
 * @since 1.22
 */
public class TGRepresentationUtils {

    public static boolean findByIdentifier(ReadGraph graph, Resource targetIndex, Resource source) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        GUID guid = graph.getPossibleRelatedValue(source, L0.identifier, GUID.BINDING);
        if(guid != null) {
            List<Resource> exist = graph.syncRequest(new QueryIndex(targetIndex, L0.Entity, "GUID:" + guid.indexString()));
            return !exist.isEmpty();
        }
        return false;
    }

    public static Function1<Statement, ExclusionDecision> computeExclusionFunction(ReadGraph graph, Resource[] resources, Map<String,Object> hints) throws DatabaseException {
        Resource targetResource = (Resource)hints.get(ClipboardUtils.HINT_TARGET_RESOURCE);
        if(targetResource != null) {
            Resource targetRoot = graph.syncRequest(new PossibleIndexRoot(targetResource));
            if(targetRoot != null) {
                for(Resource r : resources) {
                    Resource sourceRoot = graph.syncRequest(new PossibleIndexRoot(r));
                    // We are copying a complete index root => no need to change guids
                    if(r.equals(sourceRoot))
                        continue;
                    if(targetRoot.equals(sourceRoot)) {
                        // Copy into same index root => guid rewrite is needed
                        return new GUIDExclusionFunction(graph);
                    }
                    // First try the root GUID
                    if(findByIdentifier(graph, targetRoot, r))
                        return new GUIDExclusionFunction(graph);

                    // The root is OK - check everything beneath
                    Pair<List<ConsistsOfProcessEntry>,Set<Resource>> pair = ConsistsOfProcess.walk(graph, Collections.singleton(SeedSpec.internal(r)), true); 
                    List<ConsistsOfProcessEntry> entries = pair.first;
                    for(ConsistsOfProcessEntry entry : entries) {
                        if(findByIdentifier(graph, targetRoot, entry.resource))
                            return new GUIDExclusionFunction(graph);
                    }
                }
            }
        }

        return null;
    }

}
