package org.simantics.db.layer0.util;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ParametrizedPrimitiveRead;
import org.simantics.db.common.request.UnaryRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.internal.SimanticsInternal;
import org.simantics.db.procedure.Listener;
import org.simantics.db.request.Read;
import org.simantics.scl.compiler.environment.specification.EnvironmentSpecification;
import org.simantics.scl.compiler.module.repository.ImportFailureException;
import org.simantics.scl.compiler.module.repository.UpdateListener;
import org.simantics.scl.compiler.runtime.RuntimeEnvironment;
import org.simantics.scl.osgi.SCLOsgi;
import org.simantics.scl.runtime.SCLContext;

/**
 * Finds the runtime environment of a model or other index root.
 * 
 * @author Hannu Niemist&ouml;
 * @author Antti Villberg
 */
public class RuntimeEnvironmentRequest extends UnaryRead<Resource, RuntimeEnvironment> {

    public RuntimeEnvironmentRequest(Resource parameter) {
        super(parameter);
    }
    
    protected void fillEnvironmentSpecification(EnvironmentSpecification environmentSpecification) {
    }

    static class UpdateListenerImpl extends UpdateListener {
    	    	
    	final EnvironmentSpecification environmentSpecification;
    	final Listener<RuntimeEnvironment> callback;
    	
    	UpdateListenerImpl(EnvironmentSpecification environmentSpecification, Listener<RuntimeEnvironment> callback) {
    		this.environmentSpecification = environmentSpecification;
    		this.callback = callback;
    	}

        @Override
        public void notifyAboutUpdate() {
        	if(callback.isDisposed()) {
        	    stopListening();
        		return;
        	}
        	getRuntimeEnvironment(environmentSpecification, callback, this);
        }
    };     

    public static void getRuntimeEnvironment(EnvironmentSpecification environmentSpecification, Listener<RuntimeEnvironment> callback, UpdateListenerImpl listener) {

        try {
            
            SCLContext context = SCLContext.getCurrent();
            
            RuntimeEnvironment env;
            Object graph = context.get("graph");
            if(graph == null)
                try {
                    env = SimanticsInternal.getSession().syncRequest(new Read<RuntimeEnvironment>() {
                        @Override
                        public RuntimeEnvironment perform(ReadGraph graph) throws DatabaseException {
                            
                            SCLContext sclContext = SCLContext.getCurrent();
                            Object oldGraph = sclContext.get("graph");
                            try {
                                sclContext.put("graph", graph);
                                return SCLOsgi.MODULE_REPOSITORY.createRuntimeEnvironment(
                                        environmentSpecification,
                                        callback.getClass().getClassLoader(), listener);
                            } catch (ImportFailureException e) {
                                throw new DatabaseException(e);
                            } catch (Throwable t) {
                                throw new DatabaseException(t);
                            } finally {
                                sclContext.put("graph", oldGraph);
                            }
                        }
                    });
                } catch (DatabaseException e) {
                    callback.exception(e);
                    return;
                }
            else 
                env = SCLOsgi.MODULE_REPOSITORY.createRuntimeEnvironment(
                        environmentSpecification,
                        callback.getClass().getClassLoader(), listener);
            callback.execute(env);
        } catch (ImportFailureException e) {
            callback.exception(new DatabaseException(e));
        }

    }
    
    @Override
    public RuntimeEnvironment perform(ReadGraph graph)
            throws DatabaseException {
        final EnvironmentSpecification environmentSpecification = EnvironmentSpecification.of(
                "Builtin", "",
                "StandardLibrary", "",
                "Simantics/All", "");
        fillEnvironmentSpecification(environmentSpecification);
        Resource mainModule = Layer0Utils.getPossibleChild(graph, parameter, "SCLMain");
        String mainModuleUri;
        if(mainModule != null) {
            mainModuleUri = graph.getURI(mainModule);
            environmentSpecification.importModule(mainModuleUri, "");
        }
        else
            mainModuleUri = graph.getURI(parameter) + "/#"; // Add something dummy to the model uri that cannot be in a real URI
        
            return graph.syncRequest(new ParametrizedPrimitiveRead<String, RuntimeEnvironment>(mainModuleUri) {
                
                UpdateListenerImpl sclListener;
                
            	@Override
            	public void register(ReadGraph graph, Listener<RuntimeEnvironment> procedure) {

            		SCLContext context = SCLContext.getCurrent();
            		Object oldGraph = context.put("graph", graph);
            		try {

            			if(procedure.isDisposed()) {
            				getRuntimeEnvironment(environmentSpecification, procedure, null);
            			} else {
            			    sclListener = new UpdateListenerImpl(environmentSpecification, procedure);
            			    sclListener.notifyAboutUpdate();
	        			}

            		} finally {
            			context.put("graph", oldGraph);
            		}

            	}
                
                @Override
                public void unregistered() {
                	if(sclListener != null)
                	    sclListener.stopListening();
                }
                
            });
    }
    
    @Override
    public int hashCode() {
        return 31*getClass().hashCode() + super.hashCode();
    }

}
