package org.simantics.db.layer0.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.simantics.db.AsyncReadGraph;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.ResourceMap;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.SubgraphExtent.ExtentStatus;
import org.simantics.db.layer0.util.TransferableGraphConfiguration2.SeedSpec;
import org.simantics.db.layer0.util.TransferableGraphConfiguration2.SeedSpec.SeedSpecType;
import org.simantics.db.procedure.AsyncContextMultiProcedure;
import org.simantics.db.procedure.Procedure;
import org.simantics.db.service.DirectQuerySupport;
import org.simantics.layer0.Layer0;
import org.simantics.utils.datastructures.Pair;

class ConsistsOfProcess {

	final List<ConsistsOfProcessEntry> result;
	final Set<Resource> childrenWithNoName;
	final AsyncContextMultiProcedure<ConsistsOfProcessEntry, Resource> structure;
	final AsyncContextMultiProcedure<ConsistsOfProcessEntry, Resource> names;

	public static Pair<List<ConsistsOfProcessEntry>,Set<Resource>> walk(ReadGraph graph, Collection<SeedSpec> specs, boolean ignoreVirtual) throws DatabaseException {
		return walk(graph, null, specs, ignoreVirtual);
	}

	public static Pair<List<ConsistsOfProcessEntry>,Set<Resource>> walk(ReadGraph graph, ResourceMap<ExtentStatus> status, Collection<SeedSpec> specs, boolean ignoreVirtual) throws DatabaseException {
		
		Collection<ConsistsOfProcessEntry> entries = new ArrayList<>(specs.size());
		for(SeedSpec r : specs) {
			if(SeedSpecType.INTERNAL.equals(r.specType))
			    entries.add(new ConsistsOfProcessEntry(null, r.resource, true));
			else
				entries.add(new ConsistsOfProcessEntry(null, r.resource, false));
		}
    	ConsistsOfProcess process = new ConsistsOfProcess(graph, status, entries, ignoreVirtual);
    	return Pair.make(process.result, process.childrenWithNoName);
    	
    }
    
    static class ConsistsOfProcessEntry {
    	public ConsistsOfProcessEntry parent;
    	public Resource resource;
    	public boolean internal;
    	public boolean valid = true;
    	public String name = null;
    	ConsistsOfProcessEntry(ConsistsOfProcessEntry parent, Resource resource, boolean internal) {
    		this.parent = parent;
    		this.resource = resource;
    		this.internal = internal;
    	}
    }
	
    private ConsistsOfProcess(ReadGraph graph, ResourceMap<ExtentStatus> status, final Collection<ConsistsOfProcessEntry> entries, final boolean ignoreVirtual) throws DatabaseException {

		final Layer0 L0 = Layer0.getInstance(graph);
		final DirectQuerySupport dqs = graph.getService(DirectQuerySupport.class);
		
		result = new ArrayList<>();
		childrenWithNoName = new HashSet<>();
		names = dqs.compileForEachObject(graph, L0.HasName, new AsyncContextMultiProcedure<ConsistsOfProcessEntry, Resource>() {

			@Override
			public void execute(AsyncReadGraph graph, ConsistsOfProcessEntry entry, Resource nameResource) {
		
				if(status != null)
					status.put(nameResource, ExtentStatus.EXCLUDED);
				
				graph.forPossibleValue(nameResource, new Procedure<String>() {

					@Override
					public void execute(String name) {
					    if(!entry.valid) return;
					    
					    if(name == null) {
					        entry.valid = false;
					    } else if (entry.name != null) {
					        entry.valid = false;
					    } else {
					        entry.name = name;
					    }
					}

					@Override
					public void exception(Throwable t) {
						Logger.defaultLogError(t);
					}
					
				});
			}

			@Override
			public void exception(AsyncReadGraph graph, Throwable throwable) {
				Logger.defaultLogError(throwable);
			}

			@Override
			public void finished(AsyncReadGraph graph, ConsistsOfProcessEntry entry) {
			    if(entry.valid) {
			        if(entry.name != null) {
			            result.add(entry);
			        } else {
			            // This one did not have a name - not a valid internal
			            childrenWithNoName.add(entry.resource);
			        }
			    } else {
			        // Something wrong has happened. Do not treat as valid internal
			        childrenWithNoName.add(entry.resource);
			    }
			}
		});
		
		structure = dqs.compileForEachObject(graph, L0.ConsistsOf, new AsyncContextMultiProcedure<ConsistsOfProcessEntry, Resource>() {

			@Override
			public void execute(AsyncReadGraph graph, ConsistsOfProcessEntry parent, Resource child) {

				if(status != null)
					if(ExtentStatus.EXCLUDED.equals(status.get(child))) return;
				
				if(!ignoreVirtual || child.isPersistent()) {
					ConsistsOfProcessEntry entry = new ConsistsOfProcessEntry(parent, child, false);
					dqs.forEachObjectCompiled(graph, child, entry, structure);
					dqs.forEachObjectCompiled(graph, child, entry, names);
				}
				
			}

			@Override
			public void finished(AsyncReadGraph graph, ConsistsOfProcessEntry parent) {
			}

			@Override
			public void exception(AsyncReadGraph graph, Throwable throwable) {
				Logger.defaultLogError(throwable);
			}

		});
		
		graph.syncRequest(new ReadRequest() {

			@Override
			public void run(ReadGraph graph) throws DatabaseException {
				for(ConsistsOfProcessEntry entry  : entries) {
					dqs.forEachObjectCompiled(graph, entry.resource, entry, structure);
					if(entry.internal) {
						// For roots names are not processed
						dqs.forEachObjectCompiled(graph, entry.resource, entry, names);
					}
				}
			}
			
		});
		
	}
	
    
}
