/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.layer0.service.impl;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Enumeration;
import java.util.UUID;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.core.runtime.Status;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.Datatypes;
import org.simantics.databoard.serialization.Serializer;
import org.simantics.databoard.type.Datatype;
import org.simantics.db.Session;
import org.simantics.db.VirtualGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.internal.Activator;
import org.simantics.db.service.VirtualGraphSupport;
import org.simantics.db.services.ServiceInitializer;
import org.simantics.graph.db.ImportAdvisor;
import org.simantics.graph.db.TransferableGraphs;
import org.simantics.graph.representation.TransferableGraph1;

/**
 * A database client service initializer which ensures that any <code>.tg</code>
 * files defined in a bundle folder <code>virtualgraphs</code> are imported into
 * randomly named in-memory virtualgraphs for the initialized session.
 * 
 * @author Antti Villberg
 */
public class VirtualGraphInitializer implements ServiceInitializer {

	@Override
	public IStatus initialize(Session s) {
		MultiStatus result = new MultiStatus(Activator.PLUGIN_ID, 0, "VirtualGraphInitialize errors:", null);
		BundleContext context = Activator.getInstance().getContext();
		Serializer datatypeSerializer = Bindings.getSerializerUnchecked( Datatype.class );
		Datatype tgType = Datatypes.getDatatypeUnchecked(TransferableGraph1.class);
		Serializer tgSerializer = Bindings.getSerializerUnchecked( TransferableGraph1.class );
		for (final Bundle b : context.getBundles()) {
			Enumeration<?> graphs = b.getEntryPaths("virtualgraphs");
			while(graphs != null && graphs.hasMoreElements()) {
				String element = (String)graphs.nextElement();
				if(!element.endsWith(".tg"))
					continue;
				URL url = b.getEntry(element);
				try {
					InputStream stream = url.openStream();
					Datatype dt = (Datatype) datatypeSerializer.deserialize( stream );
					if(dt.equals(tgType)) {
						VirtualGraphSupport support = s.getService(VirtualGraphSupport.class);
						VirtualGraph vg = support.getMemoryPersistent(UUID.randomUUID().toString());
						TransferableGraph1 tg = (TransferableGraph1) tgSerializer.deserialize(stream);
						TransferableGraphs.importVirtualGraph(s, vg, tg, 
								new ImportAdvisor());
					}
					stream.close();
				} catch (IOException e) {
					IStatus status = new Status(IStatus.ERROR, context.getBundle().getSymbolicName(), "VirtualGraphInitializer transferable graph initialization failed for '" + url.toString() + "', see exception for details.", e);
					result.add(status);
				} catch (DatabaseException e) {
					IStatus status = new Status(IStatus.ERROR, context.getBundle().getSymbolicName(), "VirtualGraphInitializer transferable graph initialization failed for '" + url.toString() + "', see exception for details.", e);
					result.add(status);
				}
			}
		}
		if (!result.isOK()) return result;
		return Status.OK_STATUS;
	}

}
