/*******************************************************************************
 * Copyright (c) 2007 VTT Technical Research Centre of Finland and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.example;

import java.io.File;
import java.util.Arrays;

import org.simantics.databoard.Accessors;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.Datatypes;
import org.simantics.databoard.accessor.file.FileArrayAccessor;
import org.simantics.databoard.accessor.file.FileVariantAccessor;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.mutable.MutableVariant;
import org.simantics.databoard.serialization.Serializer;
import org.simantics.databoard.type.ArrayType;

public class FileAccessorExample {

	public static void main(String[] args) throws Exception {
		
		//
		// File Accessor is an access to data that resides in a file in
		// byte format.
		//
		
		// Create new binary file
		File file = File.createTempFile("variable", ".dbb");
		// Create file and open an accessor
		// Files are always variants, so the root accessor is VariantAccessor.
		FileVariantAccessor fa = Accessors.createFile(file); 
		
		// We can put any value into the file
		fa.setContentValue( Bindings.STRING, "Hello World!" );
		// Or change it to completely another type 
		fa.setContentValue( Bindings.INTEGER, 500 );
		
		// Values are normally flushed automatically
		// If special *Noflush methods are used...		
		fa.setContentValueNoflush( Bindings.DOUBLE, 99.0 );
		// then a separate cache flush is required.
		fa.flush();
		
		// Close the file
		fa.close();
		
		
		
		// Re-open the file, get an accessor
		fa = Accessors.openAccessor(file); 
		// Lets read what the type is
		System.out.println( fa.getContentType() );
		// Read the content, we know its a Double so we use an instance of DoubleBinding
		Double value = (Double) fa.getContentValue( Bindings.DOUBLE );
		System.out.println( value );
		
		
		
		// Large datasets can be accessed partially
		// It conserves memory.
		
		// Initialize the file into a Float[]
		ArrayType floatArrayType = new ArrayType( Datatypes.FLOAT );
		Binding binding = Bindings.getMutableBinding( floatArrayType );
		fa.setContentValue( binding, binding.createDefault() );
		
		// Write partially 		
		// Get a sub-accessor to content of the file.   
		FileArrayAccessor aa = (FileArrayAccessor) fa.getContentAccessor();
		// Add 1024 entries one by one
		for (int i=0; i<1024; i++) {
			float v = i * 1.5f;
			aa.addNoflush( Bindings.FLOAT, v );
		}
		// Flushing ensures bytes are moved from memory cache to disc
		// Note, reading works without flushing
		aa.flush();

		// Read partially
		System.out.print("[");
		for (int i=0; i<aa.size(); i++) {
			System.out.print( aa.get(i, Bindings.FLOAT) );
			System.out.print( ", ");
		}
		System.out.println("]");
				
		// Read all-at-once
		binding = Bindings.getBinding( float[].class );
		float data[] = (float[]) aa.getValue( binding );
		System.out.println( Arrays.toString(data) );
		
		// File handle can be closed from any of its accessors (fa and aa)
		aa.close();
		
		
		
			
		// To read the data all-at-once an accessor is not needed, you can use Serializer aswell
		
		// The root of the file is always Variant so use an instance of VariantBinding
		Serializer s = Bindings.getSerializerUnchecked( Bindings.MUTABLE_VARIANT );		
		MutableVariant variant = (MutableVariant) s.deserialize( file );
		data = (float[]) variant.getValue( binding );
		System.out.println( Arrays.toString(data) );		
		
		
		// Delete tmp file
		file.delete();		
	}
	
}

