/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.tests;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.fail;

import org.junit.Before;
import org.junit.Test;
import org.simantics.databoard.util.binary.BinaryMemory;
import org.simantics.databoard.util.binary.Blob;
import org.simantics.databoard.util.binary.RandomAccessBinary;
import org.simantics.databoard.util.binary.RandomAccessBinary.ByteSide;

public class TestBlob {

	// Blob with 0..1600 bytes
	Blob blob;
	// 10 sub blobs of 160 bytes
	Blob[] level1;
	// 10*10 sub-blobs of 16 bytes
	Blob[][] level2;

	public @Before void initData() throws Exception {
		RandomAccessBinary mem;
		mem = new BinaryMemory(1600);
//		mem = BinaryFile.tempFile(1600);
		blob = new Blob(mem);
		level1 = new Blob[10];
		level2 = new Blob[10][];
		for (int i=0; i<10; i++) {			
			level1[i] = blob.createSubBlob(i*100, 160);
			level2[i] = new Blob[10];
			for (int j=0; j<10; j++) {				
				level2[i][j] = level1[i].createSubBlob(i*10, 16);
			}
		}
	}
	
	public @Test void testSize() throws Exception {
		Blob b = blob;
		assertEquals(b.length(), 1600);
		
		for (int l=0; l<10; l++) {			
			b = level1[l];
			assertEquals(b.length(), 160);
			
			for (int l2=0; l2<10; l2++) {
				b = level2[l][l2];
				assertEquals(b.length(), 16);
			}
		}
	}
	
	public @Test void testIsolation() throws Exception {
		
		Blob b = blob;
		testIsolation(b);
		testInsertRemove(b);				
		
		for (int i=0; i<10; i++) {			
			b = level1[i];
			testIsolation(b);
			testInsertRemove(b);				
			
			for (int j=0; j<10; j++) {
				b = level2[i][j];
				testIsolation(b);	
				testInsertRemove(b);				
			}
		}
		
	}
	
	void testIsolation(Blob b) throws Exception {
		int parentCount = (int) b.getSource().length()/4;
		int c = (int) b.length()/4;
		
		// Write test		
		b.position(0);
		for (int i=0; i<c; i++) {
			b.writeInt(i^0x58);
		}

		// Read test
		b.position(0);
		for (int i=0; i<c; i++) {
			assertEquals(i^0x58, b.readInt());			
		}
		
		// Fail test (cannot read more)
		try {
			b.readInt();
			fail("Did not catch boundary exceed");
		} catch (IndexOutOfBoundsException e) {}
		
		// Write once, enlarges the blob
		b.writeInt(50);
		assertEquals(c+1, b.length()/4);
		assertEquals(parentCount+1, b.getSource().length()/4);
		
		b.setLength(c*4);
		assertEquals(c, b.length()/4);
		assertEquals(parentCount, b.getSource().length()/4);
		
	}
	
	
	public void testInsertRemove(Blob b) throws Exception {		
		int c = (int) b.length()/4;
		
		// Write test		
		b.position(0);
		for (int i=0; i<c; i++) {
			b.writeInt(i^0x53);
		}
		
		// Remove 2. index
		b.position(4);
		b.removeBytes(4, ByteSide.Left);

		// Read test
		b.position(0);
		assertEquals(0x53, b.readInt());			
		for (int i=2; i<c; i++) {
			assertEquals(i^0x53, b.readInt());			
		}
		
		// Insert 2. index
		b.position(4);
		b.insertBytes(4, ByteSide.Left);
		b.writeInt(1^0x53);
		// Read test
		b.position(0);
		for (int i=0; i<c; i++) {
			assertEquals(i^0x53, b.readInt());			
		}
	}
/*	
	public @Test void testMove1Gdata() throws IOException {
		
		File file = File.createTempFile("TestBlob", ".tmp");
		file.deleteOnExit();
		BinaryFile b = new BinaryFile(file);
		try {
		
			System.out.println("Creating 1GB file.");
			for (int i=0; i<1*1024*1024*1024/4; i++) {
				b.writeInt(i);
			}
			
			System.out.println("Adding 64 bytes at position 0");
			long time = System.nanoTime();
			b.position(0l);
			b.insertBytes(64L, ByteSide.Left);
			long elapsed = System.nanoTime() - time;
			System.out.println("Elapsed: "+elapsed+" ns");

		} finally {
			b.close();
			file.delete();
		}
	}
*/

}

