package org.simantics.structural.synchronization.base;

import gnu.trove.map.hash.THashMap;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;

import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.structural.synchronization.protocol.Connection;
import org.simantics.structural.synchronization.protocol.SerializedVariable;

abstract public class ModuleUpdaterBase<T extends ComponentBase<T>> {

    public String moduleType;
    public THashMap<String, PropertyUpdateRule<T>> propertyUpdateRules =
            new THashMap<String, PropertyUpdateRule<T>>();
    public THashMap<String, ConnectionUpdateRule<T>> connectionUpdateRules =
            new THashMap<String, ConnectionUpdateRule<T>>();
    public boolean isUserComponent;
    public boolean isComposite;
    public String subprocessType;
    
    public ModuleUpdaterBase(String moduleType) {
        this.moduleType = moduleType;
    }
    
    public void addPropertyUpdateRule(PropertyUpdateRule<T> rule) {
        propertyUpdateRules.put(rule.getPropertyName(), rule);
    }
    
    public void addConnectionUpdateRule(ConnectionUpdateRule<T> rule) {
        connectionUpdateRules.put(rule.getConnectionPointName(), rule);
    }

    public void create(ModuleUpdateContext<T> context, Collection<SerializedVariable> properties, Collection<Connection> connections) {
        context.command = createAddCommandBuilder(context.getModuleName()); 
        applyRules(context, true, properties, connections);
    }
    
    abstract public CommandBuilder createAddCommandBuilder(String name);

    public void update(ModuleUpdateContext<T> context, Collection<SerializedVariable> properties, Collection<Connection> connections) {
        // Check that the module type matches
        int moduleTypeId = context.getSolver().getModuleType(context.getModuleId());
        String moduleTypeName = context.getSolver().getName(moduleTypeId);
        if(!moduleTypeName.equals(moduleType)) {
            context.getSolver().remove(context.getModuleId());
            context.component.componentId = -1;
            context.setModuleId(-1);
            create(context, properties, connections);
        }
        
        // Update
        else {
            context.command = createUpdateCommandBuilder(context.getModuleName());
            applyRules(context, false, properties, connections);
        }
    }
    
    abstract public CommandBuilder createUpdateCommandBuilder(String name);
    
    private void applyRules(ModuleUpdateContext<T> context, boolean inCreate,
            Collection<SerializedVariable> properties, Collection<Connection> connections) {
        THashMap<String, Variant> propertyMap = new THashMap<String, Variant>(properties.size());
        Map<String, Collection<String>> connectionMap = connections.isEmpty()
                ? Collections.<String, Collection<String>>emptyMap()
                : new THashMap<String, Collection<String>>(connections.size());
        for(SerializedVariable property : properties)
            propertyMap.put(property.name, property.value);
        for(Connection connection : connections)
            connectionMap.put(connection.relation, connection.connectionPoints);

        context.incPendingCount(); // To prevent premature execution of the command
        for(SerializedVariable property : properties) {
            PropertyUpdateRule<T> rule = propertyUpdateRules.get(property.name);
            if(rule != null)
                rule.apply(context, inCreate, propertyMap, connectionMap, property.value);
            else if(property.name.equals("IsAttached"))
                ;
            else
                if(SynchronizationEventHandlerBase.TRACE_EVENTS)
                    System.out.println("    skipped property " + property.name + " " + property.toString());
        }
        if(inCreate) {
            for(Connection connection : connections) {
                ConnectionUpdateRule<T> rule = connectionUpdateRules.get(connection.relation);
                if(rule != null)
                    rule.apply(context, propertyMap, connection.connectionPoints);
                else
                	if(SynchronizationEventHandlerBase.TRACE_EVENTS)
                	    System.out.println("    skipped connection " + connection.relation + " " + connection.connectionPoints);
            }
        }
        else {
            THashMap<String, ConnectionUpdateRule<T>> unusedConnectionUpdateRules =
                    new THashMap<String, ConnectionUpdateRule<T>>(connectionUpdateRules);
            for(Connection connection : connections) {
                ConnectionUpdateRule<T> rule = unusedConnectionUpdateRules.remove(connection.relation);
                if(rule != null)
                    rule.apply(context, propertyMap, connection.connectionPoints);
                else
                    if(SynchronizationEventHandlerBase.TRACE_EVENTS)
                        System.out.println("    skipped connection " + connection.relation + " " + connection.connectionPoints);
            }
            for(ConnectionUpdateRule<T> rule : unusedConnectionUpdateRules.values())
                rule.apply(context, propertyMap, Collections.<String>emptyList());
        }
        context.decPendingCount();
    }

}
