package org.simantics.help.core;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.eclipse.help.AbstractTocProvider;
import org.eclipse.help.ITocContribution;
import org.eclipse.help.internal.HelpPlugin;
import org.eclipse.help.internal.Topic;
import org.eclipse.help.internal.entityresolver.LocalEntityResolver;
import org.eclipse.help.internal.toc.Toc;
import org.eclipse.help.internal.toc.TocContribution;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.exception.DatabaseException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

@SuppressWarnings("restriction")
public class SimanticsTocProvider extends AbstractTocProvider {

    private static DocumentBuilder builder;
    private static Document document;
    
    private static Map<String, Path> widgetReferences; 
    private static Map<String, List<Path>> tutorials;
    
    @Override
    public ITocContribution[] getTocContributions(String locale) {
        List<ITocContribution> contributions = new ArrayList<>();
        
        if (widgetReferences == null || tutorials == null) {
            try {
                Simantics.getSession().syncRequest(new ReadRequest() {
                    @Override
                    public void run(ReadGraph graph) throws DatabaseException {
                        widgetReferences = HelpUtils.collectWidgetReferencesFromSharedLibraries(graph);
                        tutorials = HelpUtils.collectHelpsFromSharedLibraries(graph);
                        
                    }
                });
            } catch (DatabaseException e) {
                e.printStackTrace();
            }
        }

        Map<String, Toc> tocs = new HashMap<>();
        
        if (widgetReferences != null) {
            for (Map.Entry<String, Path> widgetReference : widgetReferences.entrySet()) {
                String libName = widgetReference.getKey();
                Path htmlFile = widgetReference.getValue();
                Toc toc = tocs.get(libName);
                if (toc == null) {
                    Element element = getDocument().createElement("toc");
                    toc = new Toc(element); 
                    toc.setLabel(libName);
//                    toc.setTopic(libName);
//                    toc.setHref(libName);
                    tocs.put(libName, toc);
                }
                
                Topic topic = new Topic();
                //topic.setLabel(htmlFile.getFileName().toString());
                topic.setLabel("Widget Reference");
//                String href = htmlFile.toUri().toString();
                String href = htmlFile.toString().split(Activator.PLUGIN_ID)[1].substring(1);
                href = "PLUGINS_ROOT/platform:/meta/" + Activator.PLUGIN_ID + "/" + href;
                topic.setHref(href.replace("\\", "/").replace(" ", "%20"));
                
                toc.appendChild(topic);
            }
        }

        if (tutorials != null) {
            for (Map.Entry<String, List<Path>> tutorial : tutorials.entrySet()) {
                String libName = tutorial.getKey();
                Toc toc = tocs.get(libName);
                if (toc == null) {
                    Element element = getDocument().createElement("toc");
                    toc = new Toc(element); 
                    toc.setLabel(libName);
                    toc.setTopic(libName);
                    toc.setHref(libName);
                    tocs.put(libName, toc);
                }
                
                Map<String, Topic> topics = new HashMap<>();
                
                for (Path htmlFile : tutorial.getValue()) {
                    Path path = Paths.get(htmlFile.toString().split(libName)[1]);
                    Topic topic = getOrCreateTopic(topics, toc, path);
                    String href = htmlFile.toString().split(Activator.PLUGIN_ID)[1].substring(1);
                    href = "PLUGINS_ROOT/platform:/meta/" + Activator.PLUGIN_ID + "/" + href;
//                    String href = htmlFile.toUri().toString();
                    topic.setHref(href.replace("\\", "/").replace(" ", "%20"));
                }
            }
        }
        
        for (Toc toc : tocs.values()) {
            TocContribution contribution = new TocContribution();
            contribution.setLocale(locale);
            contribution.setId(toc.getLabel());
            contribution.setCategoryId("category_" + toc.getLabel());
            contribution.setPrimary(true);
            contribution.setContributorId(Activator.PLUGIN_ID);
            contribution.setExtraDocuments(new String[0]);
            contribution.setToc(toc);
            contributions.add(contribution);
        }

        
        return (ITocContribution[])contributions.toArray(new ITocContribution[contributions.size()]);
    }
    
    private static Topic getOrCreateTopic(Map<String, Topic> topics, Toc toc, Path topicPath) {
        String topicName = topicPath.getFileName().toString();
        Topic topic = topics.get(topicName);
        if (topic == null) {
            topic = new Topic();
            topic.setLabel(topicName);
            Path parentPath = topicPath.getParent();
            if (parentPath != null && parentPath.getFileName() != null) {
                Topic parentTopic = getOrCreateTopic(topics, toc, parentPath);
                parentTopic.appendChild(topic);
            } else {
                toc.appendChild(topic);
            }
            topics.put(topicName, topic);
        }
        return topic;
    }
    
    private static Document getDocument() {
        if (document == null) {
            if (builder == null) {
                try {
                    builder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
                    builder.setEntityResolver(new LocalEntityResolver());
                }
                catch (ParserConfigurationException e) {
                    String msg = "Error creating document builder"; //$NON-NLS-1$
                    HelpPlugin.logError(msg, e);
                }
            }
            document = builder.newDocument();
        }
        return document;
    }
    
    public static void clearTocCache() {
        widgetReferences.clear();
        widgetReferences = null;
        tutorials.clear();
        tutorials = null;
        HelpPlugin.getTocManager().clearCache();
    }
}
