package org.simantics.graph.representation;

import java.io.DataInput;
import java.io.DataInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.channels.ReadableByteChannel;
import java.util.ArrayList;
import java.util.List;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.Datatypes;
import org.simantics.databoard.binding.error.RuntimeDatatypeConstructionException;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.databoard.container.DataContainers;
import org.simantics.databoard.serialization.RuntimeSerializerConstructionException;
import org.simantics.databoard.serialization.Serializer;
import org.simantics.databoard.type.Datatype;


final public class TransferableGraphFileReader extends ByteFileReader {

	InputStream in = new InputStream() {

        @Override
        public int read() throws IOException {
            return getByte();
        }
        
        @Override
        public int read(byte[] b) throws IOException {
            // FIXME not correctly implemented                
            System.arraycopy(safeBytes(b.length), 0, b, 0, b.length);
            return b.length;
        }
        
        @Override
        public int read(byte[] b, int off, int len) throws IOException {
            // FIXME not correctly implemented
            System.arraycopy(safeBytes(len), 0, b, off, len);
            return len;
        }
	    
	};
	
	
	final static class InputChannel implements ReadableByteChannel {

		final private InputStream stream;
		
		public InputChannel(InputStream stream) {
			this.stream = stream;
		}
		
		@Override
		public boolean isOpen() {
			return true;
		}

		@Override
		public void close() throws IOException {
		}

		@Override
		public int read(ByteBuffer dst) throws IOException {
			int pos = dst.position();
			int limit = dst.limit();
			int i=stream.read(dst.array(), pos, limit-pos);
			//System.err.println("Read " + i + " (expected " + dst.array().length + ")");
			return i;
		}
		
	}
	
	private static boolean init = true;
	
	final private static int SIZE = 1<<18;
	final private static int HEADER = headerSize();
	final private int header;
	
	public TransferableGraphFileReader(File file) throws IOException {
		super(file, SIZE);
		if(init) {
			init=false;
			TransferableGraphFileReader r = new TransferableGraphFileReader(file, 0);
			for(int i=0;i<40000;i++) r.readTG();
		}
		this.header = HEADER;
	}
	
	public TransferableGraphFileReader(InputStream stream) throws IOException {
		super(null, new InputChannel(stream), SIZE);
		if(init) {
			init=false;
			TransferableGraphFileReader r = new TransferableGraphFileReader(stream, 0);
			for(int i=0;i<40000;i++) r.readTG();
		}
		this.header = 0;
	}

	public TransferableGraphFileReader(ReadableByteChannel channel) throws IOException {
		super(null, channel, SIZE);
		if(init) {
			init=false;
			TransferableGraphFileReader r = new TransferableGraphFileReader(channel, 0);
			for(int i=0;i<40000;i++) r.readTG();
		}
		this.header = 0;
	}

	public TransferableGraphFileReader(ReadableByteChannel channel, int size) throws IOException {
		super(null, channel, SIZE);
		this.header = 0;
	}

	public TransferableGraphFileReader(InputStream stream, int size) throws IOException {
		super(null, new InputChannel(stream), size);
		this.header = 0;
	}

	public TransferableGraphFileReader(File file, int size) throws IOException {
		super(file, size);
		this.header = HEADER;
	}

	private static int headerSize() {
		try {
			return Bindings.getSerializerUnchecked(Datatype.class).serialize(Datatypes.getDatatypeUnchecked(TransferableGraph1.class)).length;
		} catch (RuntimeSerializerConstructionException e) {
			throw new Error("Failed to determine TransferableGraph1 header size. ", e);
		} catch (RuntimeDatatypeConstructionException e) {
			throw new Error("Failed to determine TransferableGraph1 header size. ", e);
		} catch (IOException e) {
			throw new Error("Failed to determine TransferableGraph1 header size. ", e);
		}		
	}
	
	public TransferableGraph1 readTG() throws IOException {

		if(getSize() == 0) return null;
		
//		long start = System.nanoTime();

		final byte[] bytes = getBytes();
		
//		byteIndex = header;

		DataInputStream dis = new DataInputStream(in);

		// Header
		DataContainers.readHeader(dis);
		
		// Content variant data type
		Bindings.getSerializerUnchecked(Datatype.class).deserialize((DataInput)dis);

		int resourceCount = safeInt();
		
		List<Object> idcontext = new ArrayList<Object>(); 
		dis = new DataInputStream(in);
		Extensions extensions = (Extensions)Bindings.getSerializerUnchecked(Extensions.class).deserialize((DataInput)dis, idcontext);
		
		int identities = safeInt();
		Identity[] ids = new Identity[identities];

//		System.err.println("rc: " + resourceCount);
//		System.err.println("ids: " + identities);
		
//		long duration = System.nanoTime() - start;
//		System.err.println("start in " + 1e-9*duration + "s.");
//		start = System.nanoTime();
		
		for(int i=0;i<identities;i++) {
			int rid = safeInt();
			byte type = bytes[byteIndex++];
			// External
			if(type == 1) {
				
				int parent = safeInt();
				int nameLen = getDynamicUInt32();
				
				if(byteIndex+nameLen < SIZE) {
					ids[i] = new Identity(rid, new External(parent, utf(bytes, byteIndex, byteIndex + nameLen)));
					byteIndex += nameLen;
				} else {
					ids[i] = new Identity(rid, new External(parent, utf(safeBytes(nameLen), 0, nameLen)));
				}
				
			} 
			// Internal
			else if(type == 3) {
				
				int parent = safeInt();
				int nameLen = getDynamicUInt32();
				if(byteIndex+nameLen < SIZE) {
					ids[i] = new Identity(rid, new Internal(parent, utf(bytes, byteIndex, byteIndex + nameLen)));
					byteIndex += nameLen;
				} else {
					ids[i] = new Identity(rid, new Internal(parent, utf(safeBytes(nameLen), 0, nameLen)));
				}
				
			}
			// Root
			else if(type == 0) {
				int nameLen = getDynamicUInt32();
				String name = utf(safeBytes(nameLen), 0, nameLen);
				int nameLen2 = getDynamicUInt32();
				String rType = utf(safeBytes(nameLen2), 0, nameLen2);
				ids[i] = new Identity(rid, new Root(name, rType));

			} else if(type == 2) {
				throw new UnsupportedOperationException();
			}

		}

//		for(Identity id : ids) System.err.println("id: " + id);

		
//		duration = System.nanoTime() - start;
//		System.err.println("ids in " + 1e-9*duration + "s.");
//		start = System.nanoTime();

		int stmLength = safeInt();
//		System.err.println("statements: " + stmLength + " (" + byteIndex + ")");
		
		int[] statements = new int[stmLength];

		for(int stmIndex=0;stmIndex<stmLength;) {

			statements[stmIndex++] = safeInt();
			
			// Cached bytes 
			int avail = (SIZE-byteIndex) >> 2;
			int allowed = Math.min(stmLength-stmIndex, avail);
			for(int index = byteIndex, i=0;i<allowed;i++) {
				statements[stmIndex++] = ((bytes[index++]&0xff)<<24) | ((bytes[index++]&0xff)<<16) | ((bytes[index++]&0xff)<<8) | ((bytes[index++]&0xff));    				
			}
			byteIndex += allowed<<2;
			
		}

//		duration = System.nanoTime() - start;
//		System.err.println("stms in " + 1e-9*duration + "s.");
//		
//		start = System.nanoTime();

		int valueLength = safeInt();
//		System.err.println("values: " + valueLength + " (" + byteIndex + ")");

		Value[] values = new Value[valueLength]; 

		Serializer variantSerializer = Bindings.getSerializerUnchecked(Bindings.VARIANT);
		
		idcontext = new ArrayList<Object>(); 
		dis = new DataInputStream(in);
		
		for(int i=0;i<valueLength;i++) {
			int resource = safeInt();
			Variant value = (Variant)variantSerializer
			        .deserialize((DataInput)dis, idcontext);
			values[i] = new Value(resource, value);
			
			//System.err.println("read variant[" + resource + "]: " + value.toString().substring(0, Math.min(100, value.toString().length())));
			
		}

		
//		duration = System.nanoTime() - start;
//		System.err.println("values in " + 1e-9*duration + "s.");
		
		return new TransferableGraph1(resourceCount, ids, statements, values, extensions.map);

	}
	
	public static void main(String[] args) {

		try {
			
			File file = new File("c:/users/antti villberg/desktop/test.apros");
			TransferableGraphFileReader reader = new TransferableGraphFileReader(file, SIZE);
			reader = new TransferableGraphFileReader(file);
			long s = System.nanoTime();
			reader.readTG();
			long d = System.nanoTime() - s;
			System.err.println("Duration=" + 1e-9*d + "s.");
			
			
		} catch (Throwable t) {
			t.printStackTrace();
		}
		
	}

}
