package org.simantics.document.linking.wizard;

import java.io.File;
import java.lang.reflect.InvocationTargetException;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.PartInitException;
import org.simantics.Simantics;
import org.simantics.db.Resource;
import org.simantics.document.linking.report.ExportToPDF;
import org.simantics.document.linking.report.templates.ReportWriter;
import org.simantics.editors.Editors;
import org.simantics.utils.ui.ErrorLogger;
import org.simantics.utils.ui.ExceptionUtils;


/**
 * Wizard page for generating a report.
 * 
 * @author Marko Luukkainen <marko.luukkainen@vtt.fi>
 *
 */
public class ReportGeneratePage extends WizardPage {
	
	private Resource model;
	private String filename;
	private ReportWriter<?> reportWriter;
	
	
	private Label fileLabel;
	private Label reportLabel;
	private Label statusLabel;
	private Button generateButton;
	private Button showButton;
	private boolean generated = false;
	
	protected ReportGeneratePage(String pageName) {
		super(pageName,pageName,null);
		setGenerated(false);
	}
	
	public void setFilename(String filename) {
		this.filename = filename;
		setGenerated(false);
	}
	
	public void setReportWriter(ReportWriter<?> reportWriter) {
		this.reportWriter = reportWriter;
		setGenerated(false);
	}
	
	public void setModel(Resource model) {
		this.model = model;
		setGenerated(false);
	}

	@Override
	public void createControl(Composite parent) {
		Composite composite = new Composite(parent, SWT.NONE);
		composite.setLayout(new GridLayout(2,false));
		Label label = new Label(composite, SWT.NONE);
		label.setText("File:");
		fileLabel = new Label(composite, SWT.NONE);
		label = new Label(composite, SWT.NONE);
		label.setText("Report:");
		reportLabel = new Label(composite, SWT.NONE);
		label = new Label(composite, SWT.NONE);
		label.setText("Status:");
		this.statusLabel = new Label(composite, SWT.NONE);
		this.statusLabel.setText("Report has not been generated");
		generateButton = new Button(composite, SWT.PUSH);
		generateButton.setText("Generate report");
		generateButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				generate();
			}
		});
		showButton = new Button(composite, SWT.PUSH);
		showButton.setText("Show Report");
		showButton.addSelectionListener(new SelectionAdapter() {
			
			@Override
			public void widgetSelected(SelectionEvent e) {
				File file = new File(filename);
				if (file.exists() && file.canRead()) {
					try {
						Editors.openExternalEditor(file);
					} catch (PartInitException err) {
						ExceptionUtils.logAndShowError(err);
					}
				}
			}
		});
		showButton.setEnabled(false);
		updateContent();
		setControl(composite);
	}
	
	private void updateContent() {
		fileLabel.setText(filename == null ? "File has not been selected" : filename);
		reportLabel.setText(reportWriter == null ? "Report Writer has not been selected" : reportWriter.getName());
		generateButton.setEnabled(filename != null && reportWriter != null && model != null);
		showButton.setEnabled(generated);
		generateButton.setEnabled(!generated);
		if (!generated)
			statusLabel.setText("Report has not been generated");
	}
	
	public void setGenerated(boolean b) {
		generated = b;
		setPageComplete(generated);
	}
	
	private void generate() {
		generateButton.setEnabled(false);
		statusLabel.setText("Generating report");
		try {
			getWizard().getContainer().run(true, false, new IRunnableWithProgress() {
				
				@Override
				public void run(final IProgressMonitor monitor) throws InvocationTargetException,
						InterruptedException {

					ExportToPDF exportToPDF = new ExportToPDF(Simantics.getSession(), model);
					final IStatus status = exportToPDF.export(filename, reportWriter, monitor);
					Display.getDefault().asyncExec(new Runnable() {
						public void run() {
							if (!statusLabel.isDisposed()) {
								if (status.isOK())
									statusLabel.setText(status.getMessage());
								else {
									statusLabel.setText(status.getMessage());
									ExceptionUtils.logError(status.getException());
								}
								setPageComplete(true);	
								((Composite)getControl()).layout(true, true);
							}
							
						};
					});
					monitor.done();
				}
			});
		}  catch (InterruptedException err) {
			setErrorMessage("Report failed: " + err.getMessage());
			ErrorLogger.defaultLogError("Report failed.",err);
			statusLabel.setText("Report failed.");
		} catch (InvocationTargetException e) {
		    Throwable err = e.getCause();
			setErrorMessage("Report failed: " + err.getMessage());
			ErrorLogger.defaultLogError("Report failed.",err);
			statusLabel.setText("Report failed.");
		}
		setGenerated(true);
		
		showButton.setEnabled(true);
		generateButton.setEnabled(false);
		getContainer().updateButtons();
	}
	
	@Override
	public void setVisible(boolean visible) {
		if (visible) {
			updateContent();
		} else {
			setGenerated(false);
		}
		super.setVisible(visible);
	}
	
	public boolean isGenerated() {
		return generated;
	}

}
