package org.simantics.document.linking.report.templates;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.document.linking.ontology.DocumentLink;
import org.simantics.document.linking.report.Document;
import org.simantics.document.linking.report.Document.TextSize;
import org.simantics.document.linking.report.DocumentTitlePage;
import org.simantics.document.linking.report.RowContentProvider;
import org.simantics.document.linking.report.Table;
import org.simantics.document.linking.report.TableOfContents;
import org.simantics.document.linking.report.TextItem;
import org.simantics.document.linking.utils.SourceLinkUtil;

/**
 * Model information
 * ? Document 1 rev1
 *     Link1: Diagram, module, attribute information
 *     Link2: Diagram, module, attribute information
 * ? Document 1 rev2
 *     Link3: Diagram, module, attribute information
 * ? Document 1 rev3
 *     Link4: Diagram, module, attribute information
 * ? Document 2
 *     Link5: Diagram, module, attribute information
 *   
 * @author Marko Luukkainen <marko.luukkainen@vtt.fi>
 *
 */
public class DocumentStructureWriter extends IndexQueryReport {
	ReadGraph graph;
	Resource model;
	Map<Object, Object> context;
	DocumentLink sl;
	Comparator<Resource> referenceComparator;
	Comparator<Resource> parentComparator;
	SourceParentDiagramComparator diagramComparator;
	@Override
	public void sort(List<Resource> list) throws DatabaseException{
		NestedComparator<Resource> comp = new NestedComparator<Resource>();
		comp.addComparator(referenceComparator);
		comp.addComparator(parentComparator);
		Collections.sort(list, comp);
	}
	
	@Override
	public String getName() {
		return "Document Structure";
	}
	
	@Override
	public void start(ReadGraph graph, Resource model,Document writer, Map<Object, Object> context) throws Exception{
		super.start(graph, model, writer, context);
		this.context = context;
		DocumentTitlePage titlePage = writer.newElement(DocumentTitlePage.class);
		titlePage.writeTitle(graph, context);
		writer.newElement(TableOfContents.class);
		Table table = writer.newElement(Table.class,Document.TOC);
		table.addColumn("Name", 0.2);
		table.addColumn("Attribute", 0.35);
		table.addColumn("Value", 0.15);
		table.addColumn("Comment", 0.3);
		
		//lineWriter.nextPage();
		this.graph = graph;
		this.sl = DocumentLink.getInstance(graph);
		this.model = model;
		referenceComparator = new SourceReferenceComparator(graph,model);
		parentComparator = new SourceParentHierarchyComparator(graph,model);
		diagramComparator = new SourceParentDiagramComparator(graph, model);
		clearProviders();
		LinkContentProvider linkContentProvider = new LinkContentProvider();
		HierarchyContentProvider hierarchyContentProvider = new HierarchyContentProvider(linkContentProvider);
		addLineProvider(hierarchyContentProvider);
		addCellProvider(linkContentProvider);
	}

	private class HierarchyContentProvider implements RowContentProvider<Resource> {
		
		private LinkContentProvider linkContentProvider;
		
		
		
		public HierarchyContentProvider(LinkContentProvider linkContentProvider) {
			super();
			this.linkContentProvider = linkContentProvider;
		}



		@Override
		public void setText(Document writer, Resource previous,
					Resource current, Resource next, TextItem[] row) throws Exception {
			boolean writeDoc = false;
			boolean writeDiag = false;
			boolean writeToc = false;
			Table table = writer.getCurrentElement(Table.class);
			if (previous == null) {
				writeDoc = true;
				writeDiag = true;
				writeToc = true;
			} else if (writer.getCurrentLine() == 1) {
				writeDoc = true;
				writeDiag = true;
				writeToc = referenceComparator.compare(previous, current) != 0;
			} else  {
				if (referenceComparator.compare(previous, current) != 0) {
					writeDoc = true;
					writeDiag = true;
					writeToc = true;
					linkContentProvider.forceLabel = true;
				} else {
					if (diagramComparator.compare(previous, current) != 0) {
						writeDiag = true;
						linkContentProvider.forceLabel = true;
					}
				}
			}
			
			
			if (writeDoc) {
				if (writer.getAvailableLines() < 8) {
					writer.nextPage();
					table = writer.nextElement(Table.class,Document.TOC);
				}
				else if (writer.getCurrentLine() > 2)
					table = writer.nextElement(Table.class,Document.TOC);
				
				
				Resource doc2 = SourceLinkUtil.getReferredDocument(graph, current);
//				TextSize size = table.getTextSize();
//				table.setTextSize(TextSize.MEDIUM);
				TextItem label = null;
				if (doc2 != null) {
					label = getDocumentItem(doc2);
				} else {
					label = getNonExistingDocumentItem();
				}
				if (writeToc) {
					TableOfContents toc = writer.getCurrentElement(TableOfContents.class);
					toc.addTocElement(label.getText(), table);
				}
				//table.writeRow(label);
				table.setTitle(label);
//				table.setTextSize(size);
			}
			if (writeDiag) {
				if (writer.getAvailableLines() < 3)
					writer.nextPage();
				Resource parent = graph.getSingleObject(current, sl.hasSource_Inverse);
				List<Resource> path = SourceLinkUtil.getDiagramPath(graph, model, parent);
				if (path != null) {
					TextSize size = table.getTextSize();
					table.setTextSize(TextSize.MEDIUM);
					table.writeRow("  " + diagramComparator.getText(path.get(path.size()-1)));
					table.setTextSize(size);
				}
			}
				
		}
	}
	
	
	private class LinkContentProvider implements RowContentProvider<Resource> {
		
		private boolean forceLabel = false;
		
		@Override
		public void setText(Document writer, Resource previous,
				Resource source, Resource next, TextItem[] text) throws Exception {
			Resource holder = graph.getSingleObject(source, sl.hasSource_Inverse);
			if (forceLabel || previous == null || !graph.getSingleObject(previous, sl.hasSource_Inverse).equals(graph.getSingleObject(source, sl.hasSource_Inverse))) {
				String holderName = NameUtils.getSafeLabel(graph, holder);
				if (holderName.length() == 0)
					holderName = NameUtils.getSafeName(graph, holder);
				text[0] = writer.newItem(TextItem.class);
				text[0].setText(holderName);
				forceLabel = false;
			}
			if (graph.isInstanceOf(source, sl.FunctionalSource)) {
				Resource relation = graph.getPossibleObject(source, sl.consernsRelation);
				text[1] = writer.newItem(TextItem.class);
				if (relation != null) {
					String relationName = NameUtils.getSafeLabel(graph, relation);
					if (relationName.length() == 0)
						relationName = NameUtils.getSafeName(graph, relation);
					Object value = graph.getPossibleRelatedValue(holder, relation);
					text[1].setText(relationName);
					if (value != null) {
						text[2] = writer.newItem(TextItem.class);
						text[2].setText(SourceLinkUtil.getValueString(value));
					}
				} else {
					text[1].setText("Error in property reference ");
				}
			}
			
			String comment = graph.getPossibleRelatedValue(source, sl.hasSourceComment);
			if (comment != null) {
				text[3] = writer.newItem(TextItem.class);
				text[3].setText(comment);
			}
			
		}
		
	}
}