package org.simantics.export.core.pdf;

import java.awt.Graphics2D;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.simantics.Simantics;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.export.core.ExportContext;
import org.simantics.export.core.error.ExportException;
import org.simantics.export.core.intf.Format;
import org.simantics.export.core.manager.Content;
import org.simantics.utils.page.MarginUtils.Margins;
import org.simantics.utils.page.PageDesc;
import org.simantics.utils.page.PageOrientation;

import com.lowagie.text.Document;
import com.lowagie.text.DocumentException;
import com.lowagie.text.Element;
import com.lowagie.text.ExceptionConverter;
import com.lowagie.text.Font;
import com.lowagie.text.PageSize;
import com.lowagie.text.Phrase;
import com.lowagie.text.Rectangle;
import com.lowagie.text.pdf.AcroFields;
import com.lowagie.text.pdf.BadPdfFormatException;
import com.lowagie.text.pdf.ColumnText;
import com.lowagie.text.pdf.FontMapper;
import com.lowagie.text.pdf.PdfContentByte;
import com.lowagie.text.pdf.PdfCopy;
import com.lowagie.text.pdf.PdfDictionary;
import com.lowagie.text.pdf.PdfFileSpecification;
import com.lowagie.text.pdf.PdfImportedPage;
import com.lowagie.text.pdf.PdfReader;
import com.lowagie.text.pdf.PdfSignatureAppearance;
import com.lowagie.text.pdf.PdfStamper;
import com.lowagie.text.pdf.PdfTemplate;
import com.lowagie.text.pdf.PdfWriter;

/**
 * A PDF writer object.  
 *
 * @author toni.kalajainen@semantum.fi
 */
public class ExportPdfWriter {

	/** PDF Document */
	public Document document;

	/** PDF Output stream */
	public PdfCopy pdfCopy;	
	
	/** Open output stream */
	public FileOutputStream fos;

	/** The direct content byte of the document */
	public PdfContentByte cb;
	
	/** Contains Pdf Templates, e.g. symbols. Resource Uri -> Template mapping */
	public Map<String, PdfTemplate> templates = new HashMap<String, PdfTemplate>();
	
	/** Pages */
	public List<Page> pages = new ArrayList<Page>(); 
	
	/** Suggested Page Desc */
	public PageDesc defaultPageDesc;
	
	/** PageDesc as PDF rectangle */
	public Rectangle defaultRectangle;
	
	/** Initialized FontMapper */
	public FontMapper fontMapper;
	
	/** The output file */
	public File outputFile;
	
	/** All options */
	public Variant options;
	
	/** Export Context */
	public ExportContext ctx;
	
	/** The margins the user selected. */
	public Margins margins;
	
	/** Compression Level */
	public int compressionLevel;

	/**
	 * Create new page.
	 * 
	 * @param (Optional) page description. If null is used, the default size is used.
	 * @return Page for writing
	 * @throws ExportException
	 */
	public Page createPage( PageDesc pd ) throws ExportException {
				
		Rectangle rect; 
		if ( pd == null || pd.isInfinite() ) {
			pd = defaultPageDesc;
			rect = defaultRectangle;
		} else {
			rect = toRectangle( pd );
		}
		
		Page page = new Page( pd, rect, pages.size() );				
		pages.add(page);		
		return page;
	}
	
	/**
	 * Create a new template.
	 * 
	 * Note, template is not visible on the document until you add it with 
	 *   cb.addTemplate( template.tp, 0, 0);
	 *  
	 * @param name (Optional) Template identifier 
	 * @param pd (Optional) template size description. If null is used, the default size is used.
	 * @return Template handle
	 * @throws ExportException
	 */
	public Template createTemplate( String name, PageDesc pd ) throws ExportException {
		Rectangle rect; 
		if ( pd == null || pd.isInfinite() ) {
			pd = defaultPageDesc;
			rect = defaultRectangle;
		} else {
			rect = toRectangle( pd );
		}
		
        int w = (int) pd.getWidth();
        int h = (int) pd.getHeight();
        PdfTemplate tp = cb.createTemplate(w, h);        
		Template canvas = new Template( name, pd, rect, tp );
		canvas.name = name;
        if ( name!=null ) templates.put(name, tp);		
		return canvas;
	}	
	
	/**
	 * Sign the file with a private+public key pair (PPK).
	 * The file must be closed already. 
	 * 
	 * @param keystoreFile the keystore file
	 * @param keystorePassword (optional) password 
	 * @param privateKeyPassword (optional) password
	 * @param signLocation (optional) sign locaiton, e.g. "Helsinki"
	 * @param signReason (optional) e.g. "approved"
	 * @throws ExportException 
	 */
	public void sign( File keystoreFile, String keystorePassword, String privateKeyPassword, String signLocation, String signReason) throws ExportException {
		// Add Bouncycastle, if found. If not, try anyway.
		/*
        if (providerAdded.compareAndSet(false, true)) {
        	try {
        		String className = "org.bouncycastle.jce.provider.BouncyCastleProvider";
        		Class<?> clazz = Class.forName(className);
        		Provider provide = (Provider) clazz.newInstance();
        		Security.addProvider( provide );
        	} catch (SecurityException se) {
        		se.printStackTrace();
        	} catch (NullPointerException npe) {
        		npe.printStackTrace();
        	} catch (ClassNotFoundException e) {
        		e.printStackTrace();
			} catch (InstantiationException e) {
				e.printStackTrace();
			} catch (IllegalAccessException e) {
				e.printStackTrace();
			}
        }*/

        // Sign
		FileInputStream  ksfis = null;
		FileInputStream  fis = null;
		FileOutputStream fos = null;
		File signedFile = null;
		try {
			KeyStore ks = KeyStore.getInstance("pkcs12");
			signedFile = new File( outputFile.getCanonicalPath()+".signed" );
			if (signedFile.exists()) signedFile.delete();
			ksfis = new FileInputStream(keystoreFile);
			fis = new FileInputStream(outputFile);
			fos = new FileOutputStream( signedFile );
			ks.load(ksfis, keystorePassword != null ? keystorePassword.toCharArray() : null);
					
			List<String> aliases = Collections.list(ks.aliases());
			String alias = aliases.get(0);
			PrivateKey key = (PrivateKey)ks.getKey(alias, privateKeyPassword != null ? privateKeyPassword.toCharArray() : null);
			Certificate[] chain = ks.getCertificateChain(alias);

			PdfReader reader = new PdfReader( fis );
			PdfStamper stp = PdfStamper.createSignature(reader, fos, '\0');
			PdfSignatureAppearance sap = stp.getSignatureAppearance();

			/// Signature
			String fieldName = "sign"; //signReason==null?"sign":URIUtil.encodeFilename( signReason );
	        AcroFields af = stp.getAcroFields();
	        AcroFields.Item item = af.getFieldItem(fieldName);
			if (signReason!=null) sap.setReason( signReason );
			if (signLocation!=null) sap.setLocation( signLocation );
			sap.setCrypto(key, chain, null, PdfSignatureAppearance.SELF_SIGNED);
			sap.setCertificationLevel(PdfSignatureAppearance.CERTIFIED_NO_CHANGES_ALLOWED);
			sap.setRender(PdfSignatureAppearance.SignatureRenderNameAndDescription);
			
			// Make field the signature
			//sap.setVisibleSignature(fieldName);
			
			// Visible signature
	        //AcroFields af = stp.getAcroFields();
	        //AcroFields.Item item = af.getFieldItem(fieldName);
			//sap.setVisibleSignature(new Rectangle(0, 0, 100, 10), 0, fieldName);
					
			// comment next line to have an invisible signature
			//sap.setVisibleSignature(new Rectangle(682, 130, 822, 145), 1, "approved_by");
					
			//stp.getAcroFields().setField(fieldName, "someValue");
					
			stp.close();	
			reader.close();
			fis.close();
		} catch (DocumentException e) {
			throw new ExportException( e.getClass().getName()+": "+e.getMessage(), e );
		} catch (UnrecoverableKeyException e) {
			throw new ExportException( e.getClass().getName()+": "+e.getMessage(), e );
		} catch (NoSuchAlgorithmException e) {
			throw new ExportException( e.getClass().getName()+": "+e.getMessage(), e );
		} catch (CertificateException e) {
			throw new ExportException( e.getClass().getName()+": "+e.getMessage(),e  );
		} catch (IOException e) {
			throw new ExportException( e.getClass().getName()+": "+e.getMessage(), e );
		} catch (KeyStoreException e) {
			throw new ExportException( e.getClass().getName()+": "+e.getMessage(), e );
		} finally {
			if ( ksfis != null ) try { ksfis.close(); } catch (IOException e) {}
			if ( fis != null ) try { fis.close(); } catch (IOException e) {}
			if ( fos != null ) try { fos.close(); } catch (IOException e) {}
					
			if ( signedFile!=null && signedFile.exists() && outputFile.exists() ) {
				outputFile.delete();
				signedFile.renameTo( outputFile );
			}
			
		}
	}
	
	public void addAttachment(Content content) throws ExportException {
		try {
			if ( content.tmpFile == null ) throw new ExportException("Could not export "+content.filename+", null file.");
			if ( !content.tmpFile.exists() ) throw new ExportException("Could not export "+content.filename+", file not found.");
	
			Format format = ctx.eep.getFormat( content.formatId );
			//byte[] data = StreamUtil.readFully( content.tmpFile );
		    PdfDictionary fileParameter = new PdfDictionary();	    
    	    PdfFileSpecification spec = PdfFileSpecification.fileEmbedded(
   	    		pdfCopy,
   	    		content.tmpFile.getAbsolutePath(),
   	    		content.filename, 
   	    		null, 
   	    		true, 
   	    		"application/simantics/"+format.id(), 
   	    		fileParameter);
    	    
    	    pdfCopy.addFileAttachment( content.filename, spec );    	    
    	    
			
		} catch (IOException e) {
			throw new ExportException( e.getClass().getName()+": "+e.getMessage() );
		}
	}
	
	public void close() throws ExportException {
		// Flush & close
		try {
			if ( pages.isEmpty() ) {
				Page page = createPage(null);				
				Graphics2D g2d = page.createGraphics(true);
				try {
					g2d.drawString("This page is intentionally left blank.", 100, 100);
				} finally {
					g2d.dispose();
				}
			}
			
			for ( Page page : pages ) page.close();
			
			Font f = new Font(Font.HELVETICA, 8);

			int totalPages = 0;
			int currentPage = 1;
			
			for ( Page page : pages ) {
				PdfReader reader = new PdfReader( page.tmpFile.getAbsolutePath() );
				try {
			        totalPages += reader.getNumberOfPages();
				} finally {
					reader.close();
				}
			}
			
			for ( Page page : pages ) {
				PdfReader reader = new PdfReader( page.tmpFile.getAbsolutePath() );
				try {
			        int n = reader.getNumberOfPages();

			        for (int i = 1; i <= n; ++i) {
			            Rectangle pageSize = reader.getPageSizeWithRotation(i);

			            PdfImportedPage imp = pdfCopy.getImportedPage(reader, i);
			            PdfCopy.PageStamp ps = pdfCopy.createPageStamp(imp);

			            PdfContentByte over = ps.getOverContent();

			            ColumnText.showTextAligned(over, Element.ALIGN_RIGHT,
			            		new Phrase(
			            				String.format("%d / %d", currentPage++, totalPages), f),
			            				pageSize.getWidth()-12, 12, 0);
			            ps.alterContents();
			            pdfCopy.addPage(imp);
			            
			        }
				} finally {
					reader.close();
				}
	        }						
		} catch (IOException e) {
			throw new ExportException( e );
		} catch (ExceptionConverter e) {
			throw new ExportException( e );
		} catch (BadPdfFormatException e) {
			throw new ExportException( e );
		} catch (Exception e) {
			throw new ExportException( e );
		} finally {
			for ( Page page : pages ) {
				if ( page.tmpFile != null ) { page.tmpFile.delete(); page.tmpFile = null; }				
			}
			pages.clear();
			
			if ( document != null ) { document.close(); document = null; }
			if ( pdfCopy != null ) { pdfCopy.close(); pdfCopy = null; }
			if ( fos != null ) { try {
				fos.close();
			} catch (IOException e) {
				throw new ExportException(e);
			} fos = null; }
		}
	}
	
	public class Page {

		/** PDF Output stream */
		public PdfWriter pdfWriter;
		
		/** PDF Document */
		public Document document;
		
		/** Open output stream */
		public FileOutputStream fos;

		/** The direct content byte of the document */
		public PdfContentByte cb;
		
		/** Tmp-file where the page is written to */
		public File tmpFile;
		
		/** Suggested Page Desc */
		public PageDesc pageDesc;
		
		/** PageDesc as PDF rectangle */
		public Rectangle rectangle;
		
		/** Page number */
		public int pageNumber;
		
		Page(PageDesc pageDesc, Rectangle rect, int pageNumber) throws ExportException {
			try {
				this.pageDesc = pageDesc;
				this.rectangle = rect;
				this.pageNumber = pageNumber;
				this.tmpFile = Simantics.getTempfile("export.core", "pdf");
				this.fos = new FileOutputStream( tmpFile, false );			
				this.document = new Document(rectangle);			
				this.document.setPageSize( rect ); // redundant?
				this.pdfWriter = PdfWriter.getInstance(document, fos);
				this.pdfWriter.setPdfVersion(PdfWriter.PDF_VERSION_1_7);
				this.pdfWriter.setCompressionLevel( compressionLevel );
				this.pdfWriter.setPageEvent(new ServiceBasedPdfExportPageEvent());
				this.document.open();
				this.cb = this.pdfWriter.getDirectContent();
				if (!this.document.newPage()) throw new ExportException("Failed to create new page.");
			} catch (IOException e) {
				throw new ExportException( e );
			} catch (DocumentException e) {
				throw new ExportException( e );
			}
		}
		
		/**
		 * Create a graphics 2d Context that uses millimeters.
		 * 
		 * @param applyMargins top left position of margins is applied
		 * @return graphics 2d context
		 */
		public Graphics2D createGraphics(boolean applyMargins) {
            float w = rectangle.getWidth();
            float h = rectangle.getHeight();
			double pw = pageDesc.getOrientedWidth();
			double ph = pageDesc.getOrientedHeight();
			Graphics2D g2d = cb.createGraphics(w, h, fontMapper);
			
			if ( applyMargins ) {
				Margins m = pageDesc.getMargins();

	            double mw = pw - m.left.diagramAbsolute - m.right.diagramAbsolute;
	            double mh = ph - m.top.diagramAbsolute - m.bottom.diagramAbsolute;
	            double sx = m.left.diagramAbsolute;
	            double sy = m.top.diagramAbsolute;
	            
	            // Convert to points
	            mw = PageDesc.toPoints( mw );
	            mh = PageDesc.toPoints( mh );
	            sx = PageDesc.toPoints( sx );
	            sy = PageDesc.toPoints( sy );
	            
				g2d.translate(sx, sy);
			}
			
			g2d.scale(w/pw, h/ph);		
			return g2d;
		}
		
		/**
		 * Of area inside the margins, return the width of the page in millimeters. 
		 * 
		 * @return width (mm)
		 */
		public double getWidth() {
			Margins m = pageDesc.getMargins();
			return pageDesc.getOrientedWidth() - m.left.diagramAbsolute - m.right.diagramAbsolute; 
		}
		
		public double getHeight() {
			Margins m = pageDesc.getMargins();
			return pageDesc.getOrientedHeight() - m.top.diagramAbsolute - m.bottom.diagramAbsolute; 
		}
		
		/**
		 * Add attachment to this page
		 * @param content
		 * @throws ExportException
		 */
		public void addAttachment(Content content) throws ExportException {
			/*
			try {
				if ( content.tmpFile == null ) throw new ExportException("Could not export "+content.filename+", null file.");
				if ( !content.tmpFile.exists() ) throw new ExportException("Could not export "+content.filename+", file not found.");
		
				Format format = ctx.eep.getFormat( content.formatId );
				//byte[] data = StreamUtil.readFully( content.tmpFile );
			    PdfDictionary fileParameter = new PdfDictionary();	    
	    	    PdfFileSpecification spec = PdfFileSpecification.fileEmbedded(
	   	    		pdfWriter,
	   	    		content.tmpFile.getAbsolutePath(),
	   	    		content.filename, 
	   	    		null, 
	   	    		true, 
	   	    		"application/simantics/"+format.id(), 
	   	    		fileParameter);
	    	    
	    	    pdfWriter.addFileAttachment( content.filename, spec );	    	    				
			} catch (IOException e) {
				throw new ExportException( e.getClass().getName()+": "+e.getMessage() );
			}*/
			ExportPdfWriter.this.addAttachment(content);
		}
		
		public void close() throws ExportException {
			try {
				if ( document != null ) { document.close(); document = null; }
				if ( pdfWriter != null ) { pdfWriter.close(); pdfWriter = null; }
				if ( fos != null ) { fos.close(); fos = null; }
				if ( cb != null ) { cb = null; }
			} catch (IOException e) {
				throw new ExportException(e);
			}
		}
		
	}
	
	public class Template {

		/** Suggested Page Desc */
		public PageDesc pageDesc;
		
		/** PageDesc as PDF rectangle */
		public Rectangle rectangle;
		
		/** Template name */
		public String name;
		
		/** PdfTemplate */
		public PdfTemplate tp;

		Template(String name, PageDesc pd, Rectangle rect, PdfTemplate tp) {
			this.pageDesc = pd;
			this.rectangle = rect;
			this.name = name;
			this.tp = tp;
		}
		
		public Graphics2D createGraphics() {			
            double w = pageDesc.getWidth();
            double h = pageDesc.getHeight();
			return tp.createGraphics((float) w, (float) h, fontMapper);
		}
		
	}
	
    public static Rectangle toRectangle(PageDesc pageDesc) {
        String arg = PageDesc.toPoints(pageDesc.getWidth()) + " " + PageDesc.toPoints(pageDesc.getHeight());
        Rectangle r = PageSize.getRectangle(arg);

        if (PageOrientation.Landscape == pageDesc.getOrientation())
            r = r.rotate();

        // Disable inherent borders from the PDF writer.
        r.setBorder(0);

        return r;
    }
	
}
