package org.simantics.export.core.manager;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.simantics.Simantics;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.databoard.util.URIUtil;
import org.simantics.db.exception.DatabaseException;
import org.simantics.export.core.ExportContext;
import org.simantics.export.core.error.ExportException;
import org.simantics.export.core.intf.ContentType;
import org.simantics.export.core.intf.Exporter;
import org.simantics.export.core.intf.Format;
import org.simantics.export.core.util.ExportQueries;

/**
 * Exports a single content file into a tmp file.
 * 
 * The tmp file value is written to the Content object. 
 *
 * @author toni.kalajainen@semantum.fi
 */
public class ExportSingleContent extends ExportAction {
	
	public String contentUri;
	public String contentTypeId;
	public String formatId;
	public Content content;
	public File outputFile;
	
	public ExportSingleContent(Content content) throws ExportException 
	{
		if ( content.url == null || content.formatId == null || content.label == null || content.contentTypeId == null)
			throw new ExportException("Invalid input parameters in Content (null)");
		
		this.content = content;
		this.contentUri = content.url;
		this.formatId = content.formatId;
		this.contentTypeId = content.contentTypeId;		
	}
	
	public void execute(ExportContext ctx, IProgressMonitor monitor, Variant options) 
	throws ExportException
	{
		Format format = ctx.eep.getFormat( formatId );
		ContentType contentType = ctx.eep.getContentType( contentTypeId );
		Exporter[] exporters = ctx.eep.getExporters( formatId, contentTypeId );
		
		if ( exporters.length == 0 ) {
			throw new ExportException("No suitable exporter found for exporting "+contentType.label()+" to a "+format.label());
		}

		try {
			// Prefix must be at least 3 characters in length
			String prefix = "___" + URIUtil.encodeFilename(content.label);
			content.tmpFile = outputFile = File.createTempFile( prefix, URIUtil.encodeFilename(format.fileext()), Simantics.getTemporaryDirectory("export.core") );
		} catch (IOException e) {
			throw new ExportException(e);
		} 
		
		Object writer = format.createFile(ctx, outputFile, options);
		try {
			for (Exporter exporter : exporters ) {
				exporter.exportAction().export(
						Collections.singletonList( content ),
						writer, 
						ctx, 
						options,
						monitor,
						null);
			}
			
		} catch (ExportException ee) {
			format.closeFile( ctx, writer );
			writer = null;
			outputFile.delete();
			throw ee;
		} finally {
			if (writer!=null) format.closeFile( ctx, writer );
		}
		
		monitor.worked(10);
	}

	@Override
	public String label(ExportContext ctx) {
		String label;
		try {
			label = ctx.session.syncRequest( ExportQueries.label( contentUri ) );
		} catch (DatabaseException e) {
			label = "<error: "+e.getMessage()+">";
		}
		
		if ( outputFile == null ) return label;
		return label + " export to "+outputFile.getName();
	}

	@Override
	public int work(ExportContext ctx) {
		return 10;
	}

	public List<String> validate(ExportContext context, Variant options) {
		Format format = context.eep.getFormat( formatId );
		ContentType contentType = context.eep.getContentType( contentTypeId );
		Exporter[] exporters = context.eep.getExporters( formatId, contentTypeId );
		
		if ( exporters.length == 0 ) {
			return Collections.singletonList( "No suitable exporter found for exporting "+contentType.label()+" to a "+format.label() );
		}

		List<String> result = new ArrayList<String>(1);
		for (Exporter exporter : exporters) {
			try {
				result.addAll( exporter.exportAction().validate(contentUri, context, options) ) ;
			} catch (ExportException e) {
				result.add( "Could not create export action for the "+exporter.formatId()+" exporter. "+e.getMessage() );
			}
		}
		
		return result;
	}

	@Override
	public void cleanup(ExportContext ctx, IProgressMonitor progress, Variant options) throws ExportException {
		if ( this.outputFile != null ) { 
			this.outputFile.delete();
			content.tmpFile = null;			
		}
	}
		
}
