/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.service.internal;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import org.simantics.db.AsyncReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.VirtualGraph;
import org.simantics.db.WriteOnlyGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.service.TransferableGraphSupport;

/**
 * This interface is a proposition of a replacement for
 * {@link TransferableGraphSupport}. The main arguments for this interface are:
 * <ol>
 * <li>Streaming support for large data (using I/O streams)</li>
 * <li>Remaining performant with small data (using OutputStream in
 * {@link #getValue(AsyncReadGraph, Resource, OutputStream)})</li>
 * <li>Still very similar to {@link TransferableGraphSupport} and therefore
 * should be easy to implement</li>
 * </ol>
 * 
 * @author Tuukka Lehtonen
 */
public interface RawDataSupport {

    /**
     * Retrieve raw value/file data attached to the specified resource into the
     * specified output stream.
     * 
     * @param graph valid database read handle to prove the request is performed
     *        from within a read or write transaction.
     * @param resource resource to get value/file from
     * @param output stream to write the raw data into
     * @return -1 if resource contained no value/file, number of bytes written
     *         to the stream otherwise.
     * @throws IOException problems writing the output stream
     * @throws DatabaseException error occurred while writing the database
     */
    long getValue(AsyncReadGraph graph, Resource resource, OutputStream stream) throws IOException, DatabaseException;

    /**
     * Set raw value/file data attached to the specified resource from the
     * specified input stream.
     * 
     * <p>
     * Write transactions are single-threaded and exclusive which implies that a
     * client cannot invoke
     * {@link #getValue(AsyncReadGraph, Resource, OutputStream)} and
     * {@link #setValue(WriteOnlyGraph, Resource, VirtualGraph, InputStream)}
     * simultaneously in another read transaction.
     * 
     * @param graph valid database read handle to prove the request is performed
     *        from within a write transaction.
     * @param resource resource to set value/file for
     * @param provider provider for writing to virtual graphs, may be
     *        <code>null</code>
     * @param input stream to read the raw data from
     * @return number of raw data bytes written to the database.
     * @throws IOException problems reading the input stream
     * @throws DatabaseException error occurred while writing the database
     */
    long setValue(WriteOnlyGraph graph, Resource resource, VirtualGraph provider, InputStream input)
    throws IOException, DatabaseException;

}
