/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db;

import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ExternalValueException;

/**
 * This interface is for accessing external values of resources. External
 * values are values of resources which are stored outside the cluster of the
 * resource. These values support partial read and write.
 *
 * @author Kalle Kondelin
 */
public interface ExternalValueSupport {
    /**
     * Removes the value from the resource. Also truncates the value to zero
     * length so that if the value is created again it does not have any content.
     *
     * @param graph to ensure that caller has write transaction.
     * @param resource which value is to be removed.
     * @throws DatabaseException
     */
    void removeValue(WriteGraph graph, Resource resource)
    throws DatabaseException;
    /**
     * Write some or all bytes of an external value.
     *
     * IMPLEMENTATION NOTE:
     * The current implementation for the value is a file and every transaction
     * you write the value the whole file is copied and then the modification
     * is applied.
     * If you write internal value or create external value you must write the
     * whole value because the value is not initialized.
     * If you write a value that is so big that server can't hold it in memory
     * during the commit you must split it in parts by using commitAndContinue
     * method.
     *
     * @param graph to ensure that caller has write transaction.
     * @param resource which value is modified.
     * @param offset from start of the value. Must be within [0, MAX_VALUE_SIZE).
     * MAX_VALUE_SIZE = (1<<58)-2. (1<<58)-1 reserved for special move message.
     * @param length of the modified segment i.e. number of bytes to write.
     * Must be positive. If zero then the value length is truncated to offset.
     * If offset + length is greater then MAX_VALUE_SIZE then the behavior is
     * undefined.
     * @param bytes to write. Must have at least length bytes.
     * @throws DatabaseException
     */
    void writeValue(WriteGraph graph, Resource resource, long offset, int length, byte[] bytes)
    throws DatabaseException;
    /**
     * If there is an old value moves it to current change set. This means that
     * old value is lost but space is saved. Must be called before first
     * writeValue call for given change set to have any effect.
     *
     * @param graph to ensure that caller has write transaction.
     * @param resource which value is to be modified i.e. the same value as
     * given to following writeValue(s).
     */
// This breaks undo.
//    void moveValue(WriteGraph graph, Resource resource)
//    throws DatabaseException;
    /**
     * Sends commit message to server but does not end transaction. This
     * enables server to release memory used by writeValue requests.
     * Waits until most of the pending requests has been sent so that the
     * memory used by the pending requests does not grow unreasonably large.
     *
     * @param graph to ensure that caller has write transaction.
     * @param wtraits required by commit message. Usually the write request
     * which is calling this method.
     * @param resource which value is modified i.e. the same value as given to writeValue.
     */
// This breaks undo.
//    void commitAndContinue(WriteGraph graph, WriteTraits wtraits, Resource resource)
//    throws DatabaseException;
    /**
     * Read some or all bytes of an external value.
     *
     * @param graph to ensure that caller has read transaction.
     * @param resource which value is to be read.
     * @param offset from start of the value. Must be within [0, valueSize).
     * @param length of the segment to read. Offset + length must be within [0, valueSize].
     * @return the bytes of the segment that was asked.
     * @throws DatabaseException
     */
    byte[] readValue(ReadGraph graph, Resource resource, long offset, int length)
    throws DatabaseException;
    /**
     * @param graph to ensure that caller has read transaction.
     * @param resource which value is to be read.
     * @return size of the value.
     * @throws DatabaseException
     * @throws ExternalValueException if the value is not external.
     */
    long getValueSize(ReadGraph graph, Resource resource)
    throws DatabaseException, ExternalValueException;

    /**
     * Wait until there is less than limit requests in outgoing request queue.
     *
     * @param limit value for requests in request queue. Must be greater than zero.
     * @return Number of requests left in request queue.
     * @throws DatabaseException
     */
    int wait4RequestsLess(int limit)
    throws DatabaseException;
}
