/*******************************************************************************
 * Copyright (c) 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.procore.server.environment;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.Arrays;

import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IProgressMonitor;
import org.simantics.db.procore.server.environment.windows.Msi;
import org.simantics.db.procore.server.environment.windows.Product;
import org.simantics.db.procore.server.environment.windows.ProductCodes;

/**
 * @author Tuukka Lehtonen
 */
public final class ServerEnvironment {

    // http://msdn.microsoft.com/en-us/library/aa368542(v=vs.85).aspx
    private static final int ERROR_SUCCESS                 = 0;
    private static final int ERROR_SUCCESS_REBOOT_REQUIRED = 3010;

    /**
     * Checks whether the current running environment has all the necessary
     * components installed to run the ProCore database server.
     * 
     * @throws ExecutionEnvironmentException if dependencies for running the
     *         database server are not met
     */
    public static void ensureServerDependenciesMet() throws ExecutionEnvironmentException {
        ExecutionEnvironment env = ExecutionEnvironment.calculate();
        switch (env.os) {
            case WINDOWS: {
                switch (env.arch) {
                    case X86:
                        Msi.checkOneOfProductsInstalled(ProductCodes.VISUAL_CPP_2008_SP1_REDIST_X86_KB2467174);
                        break;
                    case X86_64:
                        Msi.checkProductsInstalled(ProductCodes.VISUAL_CPP_2008_SP1_REDIST_X86_KB2467174, ProductCodes.VISUAL_CPP_2008_SP1_REDIST_X64_KB2467174);
                        break;
                }
                break;
            }
        }
    }

    /**
     * Checks whether the current running environment has all the necessary
     * components installed to run the ProCore database server.
     * 
     * @throws ExecutionEnvironmentException if dependencies for running the
     *         database server are not met
     */
    public static void tryInstallDependencies(IProgressMonitor monitor) throws InstallException {
        ExecutionEnvironment env = ExecutionEnvironment.calculate();
        switch (env.os) {
            case WINDOWS: {
                switch (env.arch) {
                    case X86:
                        runInstaller(monitor, ProductCodes.VISUAL_CPP_2008_SP1_REDIST_X86_KB2467174, "VC90.2008.SP1.KB2467174.redist.x86.exe");
                        break;
                    case X86_64:
                        runInstaller(monitor, ProductCodes.VISUAL_CPP_2008_SP1_REDIST_X86_KB2467174, "VC90.2008.SP1.KB2467174.redist.x86.exe");
                        runInstaller(monitor, ProductCodes.VISUAL_CPP_2008_SP1_REDIST_X64_KB2467174, "VC90.2008.SP1.KB2467174.redist.x64.exe");
                        break;
                }
                break;
            }
        }
    }

    private static void runInstaller(IProgressMonitor monitor, Product product, String installerPath) throws InstallException {
        try {
            URL url = FileLocator.find(new URL("platform:/plugin/org.simantics.db.procore.server.environment/" + installerPath));
            URL fileUrl = FileLocator.toFileURL(url);
            final File file = new File(fileUrl.getFile());
            System.out.println(file);

            Process installer = new ProcessBuilder("cmd.exe", "/C", file.toString(), "/qb").start();
            while (true) {
                try {
                    int exitValue = installer.exitValue();
                    //System.out.println("installation done, exit code: " + exitValue);
                    switch (exitValue) {
                        case ERROR_SUCCESS:
                            return;
                        case ERROR_SUCCESS_REBOOT_REQUIRED:
                            throw new RebootRequiredException(Arrays.asList(product));
                    }
                    throw new InstallException("Installation of " + product.getDescription() + " failed with error code " + exitValue, Arrays.asList(product));
                } catch (IllegalThreadStateException e) {
//                    if (monitor.isCanceled()) {
//                        installer.destroy();
//                    }
                    // Not done yet.
                    try {
                        //System.out.println("sleeping");
                        Thread.sleep(250);
                    } catch (InterruptedException ie) {
                    }
                }
            }
        } catch (IOException e) {
            throw new InstallException(e, Arrays.asList(product));
        }
    }

    // -------------------------------------------------------------------------

    public static void main(String[] args) {
        try {
            ensureServerDependenciesMet();
        } catch (ExecutionEnvironmentException e) {
            e.printStackTrace();
        }
    }

}
