package org.simantics.db.indexing;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import org.apache.lucene.index.Term;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.NumericUtils;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.common.procedure.adapter.TransientCacheListener;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.db.service.CollectionSupport;
import org.simantics.layer0.Layer0;
import org.simantics.operation.Layer0X;

public class IndexUtils {

    public static Collection<Map<String, Object>> find(ReadGraph graph, Resource index, String filter) throws DatabaseException {
    	
        Collection<Map<String, Object>> indexResult = graph.syncRequest(new QueryIndex(index, filter), TransientCacheListener.<Collection<Map<String, Object>>>instance());

        Layer0 L0 = Layer0.getInstance(graph);
        Collection<Resource> linkedRoots = graph.syncRequest(new ObjectsWithType(index, L0.IsLinkedTo, L0.IndexRoot));
        if (linkedRoots.isEmpty())
            return indexResult;

        Collection<Map<String, Object>> result = indexResult;
        for (Resource dep : linkedRoots) {
            Collection<Map<String, Object>> linkedIndexResults = find(graph, dep, filter);
            if (linkedIndexResults.isEmpty())
                continue;
            if (result == indexResult) {
                result = new ArrayList<Map<String, Object>>(indexResult.size() + linkedIndexResults.size());
                result.addAll(indexResult);
            } else {
            }
            result.addAll(linkedIndexResults);
        }
        
        return result;
        
    }

    public static List<Resource> findResources(ReadGraph graph, Resource index, String filter) throws DatabaseException {
    	
    	List<Resource> indexResult = graph.syncRequest(new QueryIndexResources(index, filter), TransientCacheListener.<List<Resource>>instance());

        Layer0 L0 = Layer0.getInstance(graph);
        CollectionSupport coll = graph.getService(CollectionSupport.class);
        
        Collection<Resource> linkedRoots = graph.syncRequest(new ObjectsWithType(index, L0.IsLinkedTo, L0.IndexRoot));
        if (linkedRoots.isEmpty())
            return indexResult;

        List<Resource> result = indexResult;
        for (Resource dep : linkedRoots) {
            Collection<Resource> linkedIndexResults = findResources(graph, dep, filter);
            if (linkedIndexResults.isEmpty())
                continue;
            if (result == indexResult) {
            	result = coll.createList();
                result.addAll(indexResult);
            }
            result.addAll(linkedIndexResults);
        }
        
        Layer0Utils.sort(graph, result);
        return result;
        
    }
    
    public static Collection<Resource> findByName(ReadGraph graph, Resource model, String name) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        HashSet<Resource> results = new HashSet<Resource>();
        
		String search = "Name:" + name;

        for(Map<String, Object> entry : find(graph, model, search)) {
        	Resource resource = (Resource)entry.get("Resource");
            if(name.equals(graph.getPossibleRelatedValue(resource, L0.HasName, Bindings.STRING))) results.add(resource);
        }
        return results;
    }

    public static Collection<Resource> findByType(ReadGraph graph, Resource model, Resource type) throws DatabaseException {
    	
        HashSet<Resource> results = new HashSet<Resource>();
        
		String search = "Types:*" + NameUtils.getSafeName(graph, type);

        for(Map<String, Object> entry : find(graph, model, search)) {
        	Resource resource = (Resource)entry.get("Resource");
        	if(graph.isInstanceOf(resource, type)) results.add(resource);
        }
        return results;
    }

    public static Collection<Resource> findByTypeAndName(ReadGraph graph, Resource model, Resource type, String name) throws DatabaseException {
    	
        Layer0 L0 = Layer0.getInstance(graph);
        
        HashSet<Resource> results = new HashSet<Resource>();
        
		String search = "Types:*" + type + " AND Name:" + name;

        for(Map<String, Object> entry : find(graph, model, search)) {
        	Resource resource = (Resource)entry.get("Resource");
        	if(graph.isInstanceOf(resource, type) && name.equals(graph.getPossibleRelatedValue(resource, L0.HasName, Bindings.STRING))) results.add(resource);
        }
        return results;
    }
    
    public static void flushIndexCaches(IProgressMonitor progress, Session session) throws Exception {

    	MemoryIndexing mem = MemoryIndexing.getInstance(session);
    	mem.flush(progress);
    	
    }
    
    public static List<Object> list(IProgressMonitor progress, Session session, Resource indexRoot) throws Exception {
    	
    	if(progress == null) progress = new NullProgressMonitor();

    	MemoryIndexing mem = MemoryIndexing.getInstance(session);
    	Layer0X L0X = Layer0X.getInstance(session);

    	mem.flush(progress);
    	    	
    	IndexedRelationsSearcher searcher = mem.get(session, L0X.DependenciesRelation, indexRoot);
    	return searcher.doList(progress, session);
    	
    }
    
    public static Term longTerm(String key, Long value) {
    	BytesRef ref = new BytesRef();    
    	NumericUtils.longToPrefixCoded( value, 0, ref );
    	return new Term(key, ref);
    }

}
