package org.simantics.simulation.export;

import java.util.ArrayList;
import java.util.List;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.Databoard;
import org.simantics.databoard.accessor.reference.ChildReference;
import org.simantics.databoard.accessor.reference.IndexReference;
import org.simantics.databoard.binding.Binding;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.common.request.PossibleObjectWithType;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.request.PossibleModel;
import org.simantics.db.layer0.variable.RVI;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.db.request.Read;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.utils.datastructures.Pair;

/**
 * This query finds CSV item datas
 *
 * @author toni.kalajainen@semantum.fi
 */
public class CSVItemsQuery implements Read<List<CSVItemsQuery.CSVItem>> {
	
	String contentUri;
	
	/**
	 * Create query using content uris.
	 * 
	 * Label format, use ""
	 * 
	 * @param contentUris uri to Chart or Subscription
	 */
	public CSVItemsQuery(String contentUri) {
		this.contentUri = contentUri;
	}
	
	@Override
	public List<CSVItem> perform(ReadGraph graph) throws DatabaseException {
		List<CSVItem> result = new ArrayList<CSVItem>();
		Resource resource = graph.getResource(contentUri);
		
		Layer0 L0 = Layer0.getInstance(graph);
		ModelingResources MOD = ModelingResources.getInstance(graph);  
		
		List<Resource> subscriptionItems = new ArrayList<Resource>();
		
		if ( graph.isInstanceOf(resource, MOD.Subscription ) ) {
			subscriptionItems.addAll( graph.syncRequest( new ObjectsWithType(resource, L0.ConsistsOf, MOD.Subscription_Item) ) );
		} else {
			// This is probably Chart. This plugin doesn't have dependency to Chart plugin
			// We can follow HasSubscriptionItem relation anyway.
			for (Resource possibleItem : graph.getObjects( resource, L0.ConsistsOf )) {
				subscriptionItems.addAll( graph.syncRequest( new ObjectsWithType(possibleItem, L0.IsRelatedTo, MOD.Subscription_Item) ));
			}
		}
		
		for ( Resource subscriptionItem : subscriptionItems ) {
			Resource model = graph.syncRequest( new PossibleModel(subscriptionItem) );
			Resource subscription = graph.syncRequest( new PossibleObjectWithType(subscriptionItem, L0.PartOf, MOD.Subscription) );
			if (subscription == null) continue;
			Variable configuration = Variables.getPossibleConfigurationContext(graph, subscription);
			if (configuration == null) continue;
			CSVItem item = new CSVItem();
			if ( model != null) item.modelUri = graph.getURI( model );
				
   			Binding rviBinding = graph.getService(Databoard.class).getBindingUnchecked( RVI.class );
            RVI rvi = graph.getPossibleRelatedValue(subscriptionItem, MOD.Subscription_Item_VariableId, rviBinding);
            if (rvi == null) continue;        	            
			
//            Resource experiment = graph.syncRequest(new PossibleExperiment(subscriptionItem));
//            String runIdentifier = ""; // experiment.getIdentifier();
			
			item.variableReference = rvi.toPossibleString(graph, configuration);
            if (item.variableReference == null) continue;

           	item.label = removeVariablePrefixPath( item.variableReference );				
           	item.unit = graph.getPossibleRelatedValue( subscriptionItem, MOD.Subscription_Item_Unit, Bindings.STRING );
           	result.add( item );
		}
				
		return result;
	}
	
    static String removeVariablePrefixPath(String rvi) {
        // Apros-specific logics:
        //   1. remove path prefix
        //   2. change IndexReferences (i-N) into "(N+1)" Apros-style indexing
        int propIndex = rvi.indexOf('#');
        if (propIndex == -1)
            return rvi;
        int prevSlash = rvi.lastIndexOf('/', propIndex);
        if (prevSlash == -1)
            return rvi;
        Pair<String, Integer> attrKey = attributeKey(rvi, propIndex + 1);
        return rvi.substring(prevSlash + 1, propIndex + 1) + attrKey.first
                + (attrKey.second != null ? "(" + (attrKey.second + 1) + ")" : "");
    }
	
    static Pair<String, Integer> attributeKey(String key, int start) {
        int iy = key.lastIndexOf('/');
        boolean isIndexed = iy >= start;
        if (isIndexed) {
            ChildReference child = ChildReference.parsePath(key.substring(iy + 1));
            if (child instanceof IndexReference)
                return Pair.make(key.substring(start, iy), ((IndexReference) child).getIndex());
        }
        return Pair.make(key.substring(start), null);
    }
    
    public static class CSVItem {
    	public String modelUri;
    	public String label;
    	public String variableReference;
    	public String unit;
    }
    
}

