/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.layer0.utils.representation.representations;

import java.io.IOException;
import java.lang.reflect.Array;
import java.util.List;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.Databoard;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.databoard.parser.DataValuePrinter;
import org.simantics.databoard.type.Datatype;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.common.utils.OrderedSetUtils;
import org.simantics.db.exception.BindingException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.DoesNotContainValueException;
import org.simantics.db.exception.ServiceException;
import org.simantics.db.exception.ValidationException;
import org.simantics.layer0.Layer0;
import org.simantics.layer0.utils.representation.StringRepresentation2;

public class DefaultStringRepresentation2 implements StringRepresentation2 {

    private static final int MAX_ARRAY_VALUE_LENGTH_SHOWN = 32;

    Resource r;

    public DefaultStringRepresentation2(Resource r) {
        this.r = r;
    }

    @Override
    public String get(ReadGraph g) throws DatabaseException {
        return fullValueToString(g, r);
    }

    @Override
    public String get(ReadGraph g, int index) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(g);
        if (g.isInstanceOf(r, L0.OrderedSet)) {
            List<Resource> l = OrderedSetUtils.toList(g, r);
            return fullValueToString(g, l.get(index));
        }

        Object value = g.getValue(r);
        return Array.get(value, index).toString();
    }

    @Override
    public String get(ReadGraph g, int start, int size) throws DatabaseException {
        Object value = g.getValue(r);
        int valueSize = Array.getLength(value);
        int end = start+size;
        if (start > valueSize || end > valueSize)
            throw new IndexOutOfBoundsException("value size is " + valueSize + ", requested range [" + start + "-" + (end-1) + "]");

        StringBuilder b = new StringBuilder();
        boolean first = true;
        for (int i = start; i < end; ++i) {
            if (!first) {
                b.append(',');
            }
            first = false;
            b.append(Array.get(value, i));
        }
        return b.toString();
    }

    @Override
    public int getArraySize(ReadGraph g) {
        try {
            Object value = g.getValue(r);
            return Array.getLength(value);
        } catch (DatabaseException e) {
            return -1;
        }
    }

    /**
     * @param graph
     * @param resource
     * @return
     * @throws BindingException
     */
    public static String fullValueToString(ReadGraph graph, Resource resource) throws ValidationException, ServiceException, BindingException {
        String representation = null;

        // First preference is name
        final Layer0 L0 = Layer0.getInstance(graph);
        if (graph.isInstanceOf(resource, L0.Relation)) {
            representation = graph.getPossibleRelatedValue(resource, L0.HasLabel);
            if (representation == null)
                representation = graph.getPossibleRelatedValue(resource, L0.HasName);
            if (representation == null) {
                Resource inverse = graph.getPossibleInverse(resource);
                if (inverse != null) {
                    representation = graph.getPossibleRelatedValue(resource, L0.HasLabel);
                    if (representation == null)
                        representation = graph.getPossibleRelatedValue(inverse, L0.HasName);
                    if (representation != null)
                        representation = "Inverse Of " + representation;
                }
                if (representation == null) {
                    Resource single = graph.getPossibleObject(resource, L0.SubrelationOf);
                    if(single != null) {
                        String singleValue = fullValueToString(graph, single);
                        representation = "<R " + singleValue;
                    }
                }
            }
        }
        if (representation == null) {
            try {
                if (graph.isInstanceOf(resource, L0.Literal)) {
                    representation = literalStr(graph, resource);
                } else {
                    boolean first = true;
                    // Try name property/properties
                    for (Resource label : graph.getObjects(resource, L0.HasLabel)) {
                        if (!first) {
                            representation += ", ";
                        } else {
                            first = false;
                        }
                        if(graph.isInstanceOf(label, L0.Literal))
                            representation = literalStr(graph, label);
                    }
                    if (representation == null) {
                        for (Resource name : graph.getObjects(resource, L0.HasName)) {
                            if (!first) {
                                representation += ", ";
                            } else {
                                first = false;
                            }
                            representation = literalStr(graph, name);
                        }
                    }
                    if (representation == null) {
                        // Types
                        representation = ": ";
                        first = true;
                        for (Resource t : graph.getPrincipalTypes(resource)) {
                            String typeName = graph.getPossibleRelatedValue(t, L0.HasName);
                            if (typeName == null)
                                typeName = "[unnamed type]";
                            if (first)
                                first = false;
                            else
                                representation += ", ";
                            representation += typeName;
                        }
                    }
                }
            } catch (DoesNotContainValueException e) {
                throw new ValidationException(e);
            }
        }

        return representation;

    }

    static Binding dataTypeBinding = Bindings.getBindingUnchecked(Datatype.class);

    private static String literalStr(ReadGraph graph, Resource resource) throws DoesNotContainValueException,
    ValidationException, ServiceException, BindingException {

        Layer0 L0 = Layer0.getInstance(graph);

        if (graph.isInstanceOf(resource, L0.Variant)) {
            return arrayStr(graph, resource, "variant");
        } else if (graph.isInstanceOf(resource, L0.String)) {
            return arrayStr(graph, resource, "string");
        } else if (graph.isInstanceOf(resource, L0.Integer)) {
            return arrayStr(graph, resource, "integer");
        } else if (graph.isInstanceOf(resource, L0.Long)) {
            return arrayStr(graph, resource, "long");
        } else if (graph.isInstanceOf(resource, L0.Boolean)) {
            return arrayStr(graph, resource, "boolean");
        } else if (graph.isInstanceOf(resource, L0.Double)) {
            return arrayStr(graph, resource, "double");
        } else if (graph.isInstanceOf(resource, L0.Float)) {
            return arrayStr(graph, resource, "float");
        } else if (graph.isInstanceOf(resource, L0.Byte)) {
            return arrayStr(graph, resource, "byte");
        } else if (graph.isInstanceOf(resource, L0.StringArray)) {
            return arrayStr(graph, resource, "string");
        } else if (graph.isInstanceOf(resource, L0.IntegerArray)) {
            return arrayStr(graph, resource, "integer");
        } else if (graph.isInstanceOf(resource, L0.LongArray)) {
            return arrayStr(graph, resource, "long");
        } else if (graph.isInstanceOf(resource, L0.BooleanArray)) {
            return arrayStr(graph, resource, "boolean");
        } else if (graph.isInstanceOf(resource, L0.DoubleArray)) {
            return arrayStr(graph, resource, "double");
        } else if (graph.isInstanceOf(resource, L0.FloatArray)) {
            return arrayStr(graph, resource, "float");
        } else if (graph.isInstanceOf(resource, L0.ByteArray)) {
            return arrayStr(graph, resource, "byte");
        } else if (graph.isInstanceOf(resource, L0.Literal)) {
            try {
                // Print value using its datatype
                Datatype dt = graph.getPossibleRelatedValue(resource, L0.HasDataType, dataTypeBinding);
                if (dt != null) {
                    Binding dtb = Bindings.getBinding(dt);
                    //System.out.println("datatype: " + dt);
                    Object value = graph.getPossibleValue(resource, dtb);
                    if (value != null) {
                        return DataValuePrinter.writeValueSingleLine(dtb, value);
                    }
                } else {
                    return arrayStr(graph, resource, "unknown");
                }
                return "[no value: " + NameUtils.getSafeName(graph, resource, true) + "]";
            } catch (IOException e) {
                throw new ServiceException(e);
            } catch (org.simantics.databoard.binding.error.BindingException e) {
                throw new ServiceException(e);
            }
        }
        return "[unsupported literal type: " + NameUtils.getSafeName(graph, resource, true) + "]";
    }

    private static String arrayStr(ReadGraph graph, Resource resource, String type) throws DoesNotContainValueException, ValidationException, ServiceException {
        if ("variant".equals(type)) {
            try {
                Databoard db = graph.getService(Databoard.class);
                Variant variant = graph.getPossibleValue(resource, db.VARIANT);
                if (variant != null)
                    return variant.toString();
            } catch (BindingException e) {
                return "BindingException: "+e.getMessage();
            }
        }
        Object value = graph.getPossibleValue(resource);
        if (value == null)
            return "no value : " + type + "[]";
//        return value.toString();

        Class<?> vclass = value.getClass();
        boolean isArray = vclass.isArray();
        int length = 1;
        if (isArray)
            length = Array.getLength(value);

        if (!isArray)
            return value.toString();
        if (length == 1)
            return String.valueOf(Array.get(value, 0));
        if (length > MAX_ARRAY_VALUE_LENGTH_SHOWN)
            return type + "[" + length + "]";

        StringBuilder b = new StringBuilder();
        boolean first = true;
        for (int i = 0; i < length; ++i) {
            if (!first) {
                b.append(',');
            }
            first = false;
            b.append(Array.get(value, i));
        }
        return b.toString();
    }

}
