/*******************************************************************************
 * Copyright (c) 2014 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Semantum Oy - initial API and implementation
 *******************************************************************************/
package org.simantics.event.view.handler;

import java.io.File;
import java.io.FileNotFoundException;
import java.lang.reflect.InvocationTargetException;
import java.util.Calendar;

import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.handlers.HandlerUtil;
import org.osgi.service.prefs.BackingStoreException;
import org.osgi.service.prefs.Preferences;
import org.simantics.Simantics;
import org.simantics.browsing.ui.common.ErrorLogger;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.request.Read;
import org.simantics.event.Activator;
import org.simantics.event.util.EventExporter;
import org.simantics.operation.Layer0X;
import org.simantics.simulation.ontology.SimulationResource;
import org.simantics.utils.FileUtils;
import org.simantics.utils.ui.ExceptionUtils;
import org.simantics.utils.ui.dialogs.ShowError;

/**
 * @author Tuukka Lehtonen
 */
public class ExportEventsAsCsv extends AbstractHandler {

    private static final String PROP_LAST_PATH= "event.csv.export.path";

    @Override
    public Object execute(ExecutionEvent event) throws ExecutionException {
        IWorkbenchWindow window = HandlerUtil.getActiveWorkbenchWindow(event);

        try {
            String fileName = generateFileName();
            validate(window, fileName);
        } catch (DatabaseException e) {
            ErrorLogger.defaultLogError(e);
        }

        return null;
    }

    private String generateFileName() throws DatabaseException {
        String generatedName = Simantics.getSession().syncRequest(new Read<String>() {
            @Override
            public String perform(ReadGraph graph) throws DatabaseException {
                Layer0X L0X = Layer0X.getInstance(graph);
                SimulationResource SIMU = SimulationResource.getInstance(graph);
                for (Resource model : graph.syncRequest(new ObjectsWithType(Simantics.getProjectResource(), L0X.Activates, SIMU.Model))) {
                    return NameUtils.getSafeName(graph, model) + "-events-" + getTimestamp() + ".txt";
                }
                return "events-" + getTimestamp() + ".txt";
            }

            private String getTimestamp() {
                Calendar c = Calendar.getInstance();
                return c.get(Calendar.YEAR) + "-" + (1 + c.get(Calendar.MONTH)) + "-" + c.get(Calendar.DAY_OF_MONTH)
                        + "_" + c.get(Calendar.HOUR_OF_DAY) 
                        + "-" + c.get(Calendar.MINUTE) 
                        + "-" + c.get(Calendar.SECOND); 
            }
        });

        if (!FileUtils.isValidFileName(generatedName))
            generatedName = (String) Bindings.STR_VARIANT.createUnchecked(Bindings.STRING, generatedName);

        return generatedName;
    }

    public void validate(IWorkbenchWindow window, String fileName) {
        Preferences prefs = InstanceScope.INSTANCE.getNode(Activator.BUNDLE_ID);
        String lastReportPath = prefs.get(PROP_LAST_PATH, null);

        // Query for output path
        Shell parentShell = null;
        if (window != null)
            parentShell = window.getShell();

        FileDialog fd = new FileDialog(parentShell, SWT.SAVE);
        fd.setText("Select Output");
        fd.setFilterExtensions(new String[] { "*.txt", "*.csv", "*.*" });
        fd.setFilterNames(new String[] { "Tab-Separated Values (*.txt)", "Comma-Separated Values (*.csv)", "All Files (*.*)" });
        if (lastReportPath != null)
            fd.setFilterPath(lastReportPath);
        fd.setFileName(fileName);
        String path = fd.open();
        if (path != null) {
            prefs.put(PROP_LAST_PATH, path);
            try {
                prefs.flush();
            } catch (BackingStoreException e) {
                ExceptionUtils.logError(e);
            }
        } else {
            return;
        }

        String _cs = null;
        switch (fd.getFilterIndex()) {
        case 1:
            _cs = ";";
            if (!path.endsWith(".csv"))
                path += ".csv";
            break;
        case 0:
            if (!path.endsWith(".txt"))
                path += ".txt";
            // Intentional fallthrough
        default:
            _cs = "\t";
            break;
        }
        final String columnSeparator = _cs;
        final File out = new File(path);
        System.out.println("out: " + out);

        try {
            window.getWorkbench().getProgressService().busyCursorWhile(new IRunnableWithProgress() {
                @Override
                public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
                    try {
                        new EventExporter().exportCsv(monitor, out, columnSeparator);
                    } catch (Exception e) {
                        throw new InvocationTargetException(e);
                    } finally {
                        monitor.done();
                    }
                }
            });
        } catch (InvocationTargetException e) {
            Throwable t = e.getTargetException();
            if (t instanceof FileNotFoundException) {
                ShowError.showError("Export Failed", "Failed to write " + t.getMessage(), (Exception) null);
            } else {
                ExceptionUtils.logAndShowError(t);
            }
        } catch (InterruptedException e) {
            // Operation cancelled, ignore.
        }
    }

}