/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import gnu.trove.impl.hash.THash;

import java.lang.reflect.Array;

import org.simantics.db.impl.graph.ReadGraphImpl;


/**
 * An open addressed hashing implementation for Object types.
 *
 * Created: Sun Nov  4 08:56:06 2001
 *
 * @author Eric D. Friedman
 * @version $Id: UnaryQueryHash.java,v 1.2 2008/03/14 11:38:53 tuoksk Exp $
 */
abstract public class UnaryQueryHash<Procedure> extends THash {
    static final long serialVersionUID = -3461112548087185871L;

    /** the set of Objects */
    protected transient UnaryQuery<Procedure>[] _set;

    protected final UnaryQuery<Procedure> REMOVED = new UnaryQuery<Procedure>(-1) {

        @Override
        public Object computeForEach(ReadGraphImpl graph, QueryProcessor provider, Object procedure, boolean store) {
            throw new Error("Not possible.");
        }

        @Override
        public UnaryQuery<Procedure> getEntry(QueryProcessor provider) {
            throw new Error("Not possible.");
        }

        @Override
        public void putEntry(QueryProcessor provider) {
            throw new Error("Not possible.");
        }

        @Override
        public Object performFromCache(ReadGraphImpl graph, QueryProcessor provider, Procedure procedure) {
            throw new Error("Not possible.");
        }

        @Override
        public void recompute(ReadGraphImpl graph, QueryProcessor provider) {
            throw new Error("Not possible.");
        }

        @Override
        public void removeEntry(QueryProcessor provider) {
            throw new Error("Not possible.");
        }

        @Override
        public int type() {
            throw new Error("Not possible.");
        }
        
    };

    /**
     * Creates a new <code>TObjectHash</code> instance with the
     * default capacity and load factor.
     */
    public UnaryQueryHash() {
        super(DEFAULT_CAPACITY, 0.75f);
    }

    public int capacity() {
        return _set.length;
    }

    protected void removeAt(int index) {
        _set[index] = REMOVED;
        super.removeAt(index);
    }

    /**
     * initializes the Object set of this hash table.
     *
     * @param initialCapacity an <code>int</code> value
     * @return an <code>int</code> value
     */
    @SuppressWarnings("unchecked")
	protected int setUp(int initialCapacity) {
        int capacity;

        capacity = super.setUp(initialCapacity);
        _set = (UnaryQuery[])Array.newInstance(UnaryQuery.class, capacity);
        return capacity;
        
    }

    protected int index(final int id) {

        final UnaryQuery<Procedure>[] set = _set;
        final int length = set.length;
        final int hash = (31 * id) & 0x7fffffff;
        int index = hash % length;
        UnaryQuery<Procedure> cur = set[index];

        if ( cur == null ) return -1;

        // NOTE: here it has to be REMOVED or FULL (some user-given value)
        if ( cur == REMOVED || !(id == cur.id)) {
            // see Knuth, p. 529
            final int probe = 1 + (hash % (length - 2));

            do {
                index -= probe;
                if (index < 0) {
                    index += length;
                }
                cur = set[index];
            } while (cur != null
                 && (cur == REMOVED || !(id == cur.id)));
        }

        return cur == null ? -1 : index;
        
    }
    
    final protected UnaryQuery<Procedure> index2(final int id) {

        final UnaryQuery<Procedure>[] set = _set;
        final int length = set.length;
        final int hash = (31 * id) & 0x7fffffff;
        int index = hash % length;
        UnaryQuery<Procedure> cur = set[index];

        if ( cur == null ) return null;

        // NOTE: here it has to be REMOVED or FULL (some user-given value)
        if ( cur == REMOVED || (id != cur.id)) {
            // see Knuth, p. 529
            final int probe = 1 + (hash % (length - 2));

            do {
                index -= probe;
                if (index < 0) {
                    index += length;
                }
                cur = set[index];
            } while (cur != null
                 && (cur == REMOVED || (id != cur.id)));
        }

        return cur;
        
    }

    
    /**
     * Locates the index at which <tt>obj</tt> can be inserted.  if
     * there is already a value equal()ing <tt>obj</tt> in the set,
     * returns that value's index as <tt>-index - 1</tt>.
     *
     * @param obj an <code>Object</code> value
     * @return the index of a FREE slot at which obj can be inserted
     * or, if obj is already stored in the hash, the negative value of
     * that index, minus 1: -index -1.
     */
    protected int insertionIndex(final int id) {

        final UnaryQuery<Procedure>[] set = _set;
        final int length = set.length;
        final int hash = (31 * id) & 0x7fffffff;
        int index = hash % length;
        UnaryQuery<Procedure> cur = set[index];

        if (cur == null) {
            return index;       // empty, all done
        } else if (cur != REMOVED && (id == cur.id)) {
            return -index -1;   // already stored
        } else {                // already FULL or REMOVED, must probe
            // compute the double hash
            final int probe = 1 + (hash % (length - 2));

            // if the slot we landed on is FULL (but not removed), probe
            // until we find an empty slot, a REMOVED slot, or an element
            // equal to the one we are trying to insert.
            // finding an empty slot means that the value is not present
            // and that we should use that slot as the insertion point;
            // finding a REMOVED slot means that we need to keep searching,
            // however we want to remember the offset of that REMOVED slot
            // so we can reuse it in case a "new" insertion (i.e. not an update)
            // is possible.
            // finding a matching value means that we've found that our desired
            // key is already in the table
            if (cur != REMOVED) {
                // starting at the natural offset, probe until we find an
                // offset that isn't full.
                do {
                    index -= probe;
                    if (index < 0) {
                        index += length;
                    }
                    cur = set[index];
                } while (cur != null
                         && cur != REMOVED
                         && ! (id == cur.id));
            }

            // if the index we found was removed: continue probing until we
            // locate a free location or an element which equal()s the
            // one we have.
            if (cur == REMOVED) {
                int firstRemoved = index;
                while (cur != null
                       && (cur == REMOVED || ! (id == cur.id))) {
                    index -= probe;
                    if (index < 0) {
                        index += length;
                    }
                    cur = set[index];
                }
                // NOTE: cur cannot == REMOVED in this block
                return (cur != null) ? -index -1 : firstRemoved;
            }
            // if it's full, the key is already stored
            // NOTE: cur cannot equal REMOVE here (would have retuned already (see above)
            return (cur != null) ? -index -1 : index;
        }
    }

    protected int insertionIndex2(final int id, final UnaryQuery<Procedure>[] set) {

        final int length = set.length;
        final int hash = (31 * id) & 0x7fffffff;
        int index = hash % length;
        UnaryQuery<Procedure> cur = set[index];

        if (cur == null) {
            return index;       // empty, all done
        } else if (cur != REMOVED && (id == cur.id)) {
            return -index -1;   // already stored
        } else {                // already FULL or REMOVED, must probe
            // compute the double hash
            final int probe = 1 + (hash % (length - 2));

            // if the slot we landed on is FULL (but not removed), probe
            // until we find an empty slot, a REMOVED slot, or an element
            // equal to the one we are trying to insert.
            // finding an empty slot means that the value is not present
            // and that we should use that slot as the insertion point;
            // finding a REMOVED slot means that we need to keep searching,
            // however we want to remember the offset of that REMOVED slot
            // so we can reuse it in case a "new" insertion (i.e. not an update)
            // is possible.
            // finding a matching value means that we've found that our desired
            // key is already in the table
            if (cur != REMOVED) {
                // starting at the natural offset, probe until we find an
                // offset that isn't full.
                do {
                    index -= probe;
                    if (index < 0) {
                        index += length;
                    }
                    cur = set[index];
                } while (cur != null
                         && cur != REMOVED
                         && ! (id == cur.id));
            }

            // if the index we found was removed: continue probing until we
            // locate a free location or an element which equal()s the
            // one we have.
            if (cur == REMOVED) {
                int firstRemoved = index;
                while (cur != null
                       && (cur == REMOVED || ! (id == cur.id))) {
                    index -= probe;
                    if (index < 0) {
                        index += length;
                    }
                    cur = set[index];
                }
                // NOTE: cur cannot == REMOVED in this block
                return (cur != null) ? -index -1 : firstRemoved;
            }
            // if it's full, the key is already stored
            // NOTE: cur cannot equal REMOVE here (would have retuned already (see above)
            return (cur != null) ? -index -1 : index;
        }
    }

    /**
     * Convenience methods for subclasses to use in throwing exceptions about
     * badly behaved user objects employed as keys.  We have to throw an
     * IllegalArgumentException with a rather verbose message telling the
     * user that they need to fix their object implementation to conform
     * to the general contract for java.lang.Object.
     *
     * @param o1 the first of the equal elements with unequal hash codes.
     * @param o2 the second of the equal elements with unequal hash codes.
     * @exception IllegalArgumentException the whole point of this method.
     */
    protected final void throwObjectContractViolation(Object o1, Object o2)
        throws IllegalArgumentException {
        throw new IllegalArgumentException("Equal objects must have equal hashcodes. "
                                           + "During rehashing, Trove discovered that "
                                           + "the following two objects claim to be "
                                           + "equal (as in java.lang.Object.equals()) "
                                           + "but their hashCodes (or those calculated by "
                                           + "your TObjectHashingStrategy) are not equal."
                                           + "This violates the general contract of "
                                           + "java.lang.Object.hashCode().  See bullet point two "
                                           + "in that method's documentation. "
                                           + "object #1 =" + o1
                                           + "; object #2 =" + o2);
    }
} // TObjectHash
