/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import gnu.trove.procedure.TIntProcedure;
import gnu.trove.set.hash.TIntHashSet;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.concurrent.Semaphore;

import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.impl.procedure.IntProcedureAdapter;
import org.simantics.db.impl.procedure.InternalProcedure;
import org.simantics.db.procedure.ListenerBase;

final public class PrincipalTypes extends CollectionUnaryQuery<IntProcedure> {

//	public ArrayList<IntProcedure> procs = null;

	private PrincipalTypes(final int resource) {
		super(resource);
	}

	final static PrincipalTypes entry(final QueryProcessor provider, final int r) {
		return (PrincipalTypes)provider.principalTypesMap.get(r);
	}

	final static void runner(ReadGraphImpl graph, final int r, final CacheEntry parent, final ListenerBase listener, final IntProcedure procedure) {

    	QueryProcessor processor = graph.processor;
		
		PrincipalTypes entry = (PrincipalTypes)processor.principalTypesMap.get(r); 
		if(entry == null) {

			entry = new PrincipalTypes(r);
        	entry.setPending();
        	entry.clearResult(processor.querySupport);
        	entry.putEntry(processor);
			
        	processor.performForEach(graph, entry, parent, listener, procedure);
			
		} else {
			
            if(entry.isPending()) {
                synchronized(entry) {
                    if(entry.isPending()) {
                        throw new IllegalStateException();
//						if(entry.procs == null) entry.procs = new ArrayList<IntProcedure>(1);
//						entry.procs.add(procedure);
//						processor.registerDependencies(graph, entry, parent, listener, procedure, false);
//						return;
					}
				}
			}
            processor.performForEach(graph, entry, parent, listener, procedure);
		}

	}

	final public static void queryEach(ReadGraphImpl graph, final int r, final QueryProcessor provider, final CacheEntry parent, final ListenerBase listener, final IntProcedure procedure) {

		assert(r != 0);

        if(parent == null && listener == null) {
        	PrincipalTypes.computeForEach(graph, r, null, graph.processor, procedure);
        } else {
        	runner(graph, r, parent, listener, procedure);
        }

	}

	@Override
	public UnaryQuery<IntProcedure> getEntry(QueryProcessor provider) {
		return provider.principalTypesMap.get(id);
	}

	@Override
	public void putEntry(QueryProcessor provider) {
		provider.principalTypesMap.put(id, this);
	}

	@Override
	final public void removeEntry(QueryProcessor provider) {
		provider.principalTypesMap.remove(id);
	}

	static class Koss {

		private TIntHashSet set = null;
		public int single = 0;

		public boolean add(int val) {
			if(single == val) return false;
			if(single == 0) {
				single = val;
				return true;
			}
			if(set == null) set = new TIntHashSet(4);
			set.add(val);
			return true;
		}

		public int size() {

			if(single == 0) return 0;
			if(set == null) return 1;
			return set.size() + 1;

		}

		public int[] toArray() {

			int[] result = Arrays.copyOf(set.toArray(), set.size() + 1);
			result[set.size()] = single;
			return result;

		}

		public void forEach(TIntProcedure proc) {
			proc.execute(single);
			if(set != null) set.forEach(proc);
		}

	}

	@Override
	public Object computeForEach(final ReadGraphImpl procedureGraph, final QueryProcessor provider, final IntProcedure proc, final boolean store) {

		return computeForEach(procedureGraph, id, this, provider, proc);
		
	}
	
	public static Object computeForEach(final ReadGraphImpl graph, final int id, final PrincipalTypes entry, final QueryProcessor provider, final IntProcedure proc) {
		
		provider.querySupport.ensureLoaded(graph, id);
		assert(id != 0);

		int ret = provider.querySupport.getSingleInstance(id);
		if(ret > 0) {
			if(entry != null) {
				entry.add(ret);
				entry.finish(graph, provider);
			}
			proc.execute(graph, ret);
			proc.finished(graph);
			return ret;
		}

		final int instanceOf = provider.getInstanceOf();
		final int inherits = provider.getInherits();
		final int subrelationOf = provider.getSubrelationOf();

		final Koss indirect = new Koss();
		final Koss material = new Koss();

		IntProcedure directProc = new IntProcedure() {

			@Override
			public void execute(ReadGraphImpl graph, int i) {
				material.add(i);
			}

			@Override
			public void exception(ReadGraphImpl graph, Throwable t) {
				proc.exception(graph, t);
			}

			@Override
			public void finished(ReadGraphImpl graph) {
			}

		};

		IntProcedure indirectProc = new IntProcedure() {

			@Override
			public void execute(ReadGraphImpl graph, int i) {
				indirect.add(i);
			}

			@Override
			public void exception(ReadGraphImpl graph, Throwable t) {
				proc.exception(graph, t);
			}

			@Override
			public void finished(ReadGraphImpl graph) {
			}

		};

		provider.querySupport.getObjects(graph, id, instanceOf, directProc);
		provider.querySupport.getObjects(graph, id, inherits, indirectProc);
		provider.querySupport.getObjects(graph, id, subrelationOf, indirectProc);

		if(indirect.size() == 0) {
			int size = material.size(); 
			if(size == 0) {
				if(entry != null) entry.finish(graph, provider);
				proc.finished(graph);
				return null;
			} else if(size == 1) {
				int single = material.single; 
				if(entry != null) {
					entry.add(single);
					entry.finish(graph, provider);
				}
				proc.execute(graph, single);
				proc.finished(graph);
				return single;
			} else {
				addPrincipalType(graph, new TIntHashSet(4), material.toArray(), 0, provider, entry, proc);
				return null;
			}
		}

//		final AtomicInteger finishes = new AtomicInteger(0);

		indirect.forEach(new TIntProcedure() {

			int finishes = 0;
			
			@Override
			public boolean execute(final int arg0) {

				// No self-loops!
				if(arg0 == id) {
					int current = (++finishes);
					if(current == indirect.size()) {
						int size = material.size(); 
						if(size == 0) {
							if(entry != null) entry.finish(graph, provider);
							proc.finished(graph);
							return true;
						} else if(size == 1) {
							int single = material.single; 
							if(entry != null) {
								entry.add(single);
								entry.finish(graph, provider);
							}
							proc.execute(graph, single);
							proc.finished(graph);
							return true;
						} else {
							addPrincipalType(graph, new TIntHashSet(4), material.toArray(), 0, provider, entry, proc);
							return true;
						}
					}
					return true;
				}

				PrincipalTypes.queryEach(graph, arg0, provider, entry, null, new IntProcedure() {

					@Override
					public void execute(ReadGraphImpl graph, int i) {
						synchronized(material) {
							material.add(i);
						}
					}

					@Override
					public void finished(ReadGraphImpl graph) {

						int current = (++finishes);
						if(current == indirect.size()) {
							int size = material.size(); 
							if(size == 0) {
								if(entry != null) entry.finish(graph, provider);
								proc.finished(graph);
								return;
							} else if(size == 1) {
								int single = material.single; 
								if(entry != null) {
									entry.add(single);
									entry.finish(graph, provider);
								}
								proc.execute(graph, single);
								proc.finished(graph);
								return;
							} else {
								addPrincipalType(graph, new TIntHashSet(4), material.toArray(), 0, provider, entry, proc);
								return;
							}
						}

					}

					@Override
					public void exception(ReadGraphImpl graph, Throwable t) {
						proc.exception(graph, t);
					}

				});

				return true;

			}

		});
		
		return null;

	}

	private static void finish(ReadGraphImpl graph, final TIntHashSet rejects, final int[] material, final PrincipalTypes entry, final QueryProcessor provider, final IntProcedure proc) {

		if(entry != null) {
			for(int i : material) {
				if(!rejects.contains(i)) {
					entry.add(i);
				}
			}
			entry.finish(graph, provider);
		}

		for(int i : material) {
			if(!rejects.contains(i)) {
				proc.execute(graph, i);
			}
		}

		proc.finished(graph);

		return;

	}

	private static void addPrincipalType(final ReadGraphImpl graph, final TIntHashSet rejects, final int[] material, int index, final QueryProcessor provider, final PrincipalTypes entry, final IntProcedure proc) {

		//        if((counter++ % 500) == 0) System.out.println("PT " + counter + " mat = " + material.length);

		if(index == material.length) { 
			finish(graph, rejects, material, entry, provider, proc);
			return;
		}

		int type = material[index++];
		while(rejects.contains(type)) {
			if(index == material.length) { 
				finish(graph, rejects, material, entry, provider, proc);
				return;
			}
			type = material[index++];
		}
		final int nextIndex = index;

		SuperTypes.queryEach(graph, type, provider, entry, null, new InternalProcedure<IntSet>() {

			@Override
			public void execute(ReadGraphImpl graph, IntSet supers) {

				synchronized(rejects) {

					supers.forEach(new TIntProcedure() {

						@Override
						public boolean execute(int arg0) {
							rejects.add(arg0);
							return true;
						}

					});

				}

				addPrincipalType(graph, rejects, material, nextIndex, provider, entry, proc);

			}

			@Override
			public void exception(ReadGraphImpl graph, Throwable t) {
				proc.exception(graph, t);
			}

		});

	}

	@Override
	public String toString() {
		return "PrincipalTypes[" + id + "]";
	}

	final private void add(int val) {
		assert(isPending());
		IntArray v = (IntArray)getResult();
		v.add(val);
	}

	final private void finish(ReadGraphImpl graph, QueryProcessor provider) {

		assert(isPending());

//		ArrayList<IntProcedure> p = null;

		synchronized(this) {

			setReady();
//			p = procs;
//			procs = null;

		}

//		if(p != null) {
//
//			IntArray v = (IntArray)getResult();
//			if(v != null) {
//				if(v.data == null) {
//					if(v.sizeOrData != IntArray.NO_DATA) {
//						for(IntProcedure proc : p) proc.execute(graph, v.sizeOrData);
//					}
//				} else {
//					for(IntProcedure proc : p) {
//						for(int i = 0;i < v.sizeOrData ; i++) proc.execute(graph, v.data[i]);
//					}
//				}
//			}
//
//			for(IntProcedure proc : p) proc.finished(graph);
//
//		}


	}

	@Override
	public Object performFromCache(ReadGraphImpl graph, QueryProcessor provider, final IntProcedure procedure) {

		assert(isReady());

    	if(handleException(graph, procedure)) return EXCEPTED;
		
		final IntArray value = (IntArray)getResult();
		if(value.data == null) {
			if(value.sizeOrData != IntArray.NO_DATA) procedure.execute(graph, value.sizeOrData);
		} else {
			for(int i = 0;i < value.sizeOrData ; i++) procedure.execute(graph, value.data[i]);
		}

		procedure.finished(graph);
		
		return getResult();

	}

	@Override
	public void recompute(ReadGraphImpl graph, QueryProcessor provider) {

		final Semaphore s = new Semaphore(0);

		computeForEach(graph, provider, new IntProcedureAdapter() {

			@Override
			public void finished(ReadGraphImpl graph) {
				s.release();
			}

			@Override
			public void exception(ReadGraphImpl graph, Throwable t) {
				s.release();
				new Error("Error in recompute.", t).printStackTrace();
			}

		}, true);

    	while(!s.tryAcquire()) {
    		provider.resume(graph);
    	}

	}


    @Override
    boolean isImmutable(ReadGraphImpl graph) {
    	return graph.processor.isImmutable(id);
    }
    
    @Override
    protected void fillImpliedParents(QueryProcessor processor, ArrayList<CacheEntry> result) {
//		for(Objects o : Objects.entries(processor, id)) result.add(o);
    }

}
