/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import java.lang.reflect.Array;
import java.util.ArrayList;



/**
 * An implementation of the Map interface which uses an open addressed
 * hash table to store its contents.
 *
 * Created: Sun Nov  4 08:52:45 2001
 *
 * @author Eric D. Friedman
 * @version $Id: BinaryQueryHashMap.java,v 1.2 2008/03/14 11:38:53 tuoksk Exp $
 */
public class DoubleKeyQueryHashMap<Procedure> extends DoubleKeyQueryHash<Procedure> {

    /**
     * Creates a new <code>THashMap</code> instance with the default
     * capacity and load factor.
     */
    public DoubleKeyQueryHashMap() {
        super();
    }

    /**
     * initialize the value array of the map.
     *
     * @param initialCapacity an <code>int</code> value
     * @return an <code>int</code> value
     */
    protected int setUp(int initialCapacity) {
        int capacity;
        
        capacity = super.setUp(initialCapacity);
//        _values = (Object[]) new Object[capacity];
        return capacity;
    }
    
    private int sizeInternal = 0;
    
    /**
     * Inserts a key/value pair into the map.
     *
     * @param key an <code>Object</code> value
     * @param value an <code>Object</code> value
     * @return the previous value associated with <tt>key</tt>,
     * or null if none was found.
     */
    @SuppressWarnings("unchecked")
	public DoubleKeyValueMap put(final int id, final DoubleKeyValueMap value) {
    	DoubleKeyValueMap previous = null;
        Object oldKey;
        int index = insertionIndex(id);
        boolean isNewMapping = true;
        if (index < 0) {
            index = -index -1;
            previous = _set[index];
            isNewMapping = false;
        }
        oldKey = _set[index];
        _set[index] = value;
        if (isNewMapping) {
            postInsertHook(oldKey == null);
        }

        return previous;
    }
    
    public CacheEntry put(final long id, BinaryQuery value) {
    	int r1 = r1(id);
    	DoubleKeyValueMap map = get(r1);
    	if(map == null) {
    		map = new DoubleKeyValueMap(r1);
    		put(r1, map);
    	}
    	CacheEntry old = map.put(id, value);
    	if(old == null) sizeInternal++;
    	return old;
    }

    /**
     * rehashes the map to the new capacity.
     *
     * @param newCapacity an <code>int</code> value
     */
	protected void rehash(int newCapacity) {
        
        int oldCapacity = _set.length;
        DoubleKeyValueMap<Procedure> oldKeys[] = _set;

        DoubleKeyValueMap<Procedure> newKeys[]  = (DoubleKeyValueMap<Procedure>[])Array.newInstance(DoubleKeyValueMap.class, newCapacity);

        for (int i = oldCapacity; i-- > 0;) {
            if(oldKeys[i] != null && oldKeys[i] != REMOVED) {
            	DoubleKeyValueMap<Procedure> o = oldKeys[i];
                int index = insertionIndex2(o.id, newKeys);
                if (index < 0) {
                    throwObjectContractViolation(newKeys[(-index -1)], o);
                }
                newKeys[index] = o;
            }
        }
        
        _set = newKeys;
        
    }

    /**
     * retrieves the value for <tt>key</tt>
     *
     * @param key an <code>Object</code> value
     * @return the value of <tt>key</tt> or null if no such mapping exists.
     */
    final public DoubleKeyValueMap<Procedure> get(final int id) {
//        int index = index(id);
//        return index < 0 ? null : _set[index];
    	return index2(id);
    }
    
    final protected static long id(long r1, long r2) {
        long result = (r1<<32) | (r2 & 0xffffffffL); 
        return result;
    }

    final public int r1(final long id) {
        return (int)(id>>>32);
    }
    
    final public int r2(final long id) {
        return (int)id;
    }
    
    final public BinaryQuery get(final int r1, final int r2) {
    	DoubleKeyValueMap<Procedure> map = get(r1);
    	if(map == null) return null;
    	return map.get(id(r1,r2));
    }

    final public BinaryQuery get(final long id) {
    	DoubleKeyValueMap<Procedure> map = get(r1(id));
    	if(map == null) return null;
    	return map.get(id);
    }

    /**
     * Deletes a key/value pair from the map.
     *
     * @param key an <code>Object</code> value
     * @return an <code>Object</code> value
     */
    public Object remove(final int id) {
        DoubleKeyValueMap<Procedure> prev = null;
        int index = index(id);
        if (index >= 0) {
            prev = _set[index];
            removeAt(index);    // clear key,state; adjust size
            sizeInternal-=prev.size();
        }
        return prev;
    }

    public Object remove(final long id) {
    	int r1 = r1(id);
    	DoubleKeyValueMap<Procedure> map = get(r1);
    	if(map == null) return null;
    	Object removed = map.remove(id);
    	if(removed != null) sizeInternal--;
    	if(map.isEmpty()) remove(r1);
    	return removed;
    }

    final public void values(int level, CacheCollectionResult result) {
    	
        for (int i = _set.length; i-- > 0;) {
            if(_set[i] != null && _set[i] != REMOVED) {
            	DoubleKeyValueMap map = _set[i];
            	map.values(level, result);
            }
        }
    	
    }

    final public ArrayList<CacheEntry> values() {

    	ArrayList<CacheEntry> result = new ArrayList<CacheEntry>();
    	
        for (int i = _set.length; i-- > 0;) {
            if(_set[i] != null && _set[i] != REMOVED) {
            	DoubleKeyValueMap map = _set[i];
            	result.addAll(map.values());
            }
        }
    
    	return result;
    	
    }
    
    @Override
    public int size() {
        return sizeInternal;
    }
    
    final public static ArrayList<BinaryQuery> NO_VALUES = new ArrayList<BinaryQuery>();

    final public <T extends BinaryQuery> ArrayList<T> values(int r1) {

    	DoubleKeyValueMap map = get(r1);
    	if(map == null) return (ArrayList<T>)NO_VALUES;
    	
    	return map.values();
    	
    }
    
} // THashMap
