/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.mapping.rule.instructions;

import gnu.trove.map.hash.TIntIntHashMap;
import gnu.trove.procedure.TIntProcedure;
import gnu.trove.set.hash.TIntHashSet;

import org.simantics.db.ReadGraph;
import org.simantics.db.WriteGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.request.Read;
import org.simantics.layer0.utils.triggers.IModification;

public class QueryRuleInstruction implements IRuleInstruction {

	IRuleInstruction rule;
	int[] variables;
	int workSpace;

	public QueryRuleInstruction(IRuleInstruction rule) {
		this.rule = rule;
		TIntHashSet reads = new TIntHashSet();
		TIntHashSet writes = new TIntHashSet();
		rule.collectVariables(reads, writes);
		reads.removeAll(writes.toArray());
		variables = reads.toArray();
		final TIntIntHashMap map = new TIntIntHashMap();
		for(int i = 0;i<variables.length;++i)
			map.put(variables[i], i);
		workSpace = variables.length;
		writes.forEach(new TIntProcedure() {

			@Override
			public boolean execute(int arg0) {
				map.put(arg0, workSpace++);
				return true;
			}
			
		});
		rule.mapVariables(map);
	}

	@Override
	public void collectVariables(TIntHashSet reads, TIntHashSet writes) {
		reads.addAll(variables);
	}

	class Query implements Read<IModification> {

		Object[] parameters;
		
		public Query(Object[] parameters) {
			this.parameters = parameters;
		}

		QueryRuleInstruction parent() {
			return QueryRuleInstruction.this;
		}
		
		@Override
		public boolean equals(Object other) {
			if(this == other)
				return true;
			if(other == null || other.getClass() != this.getClass())
				return false;
			Query q = (Query)other;
			if(!parent().equals(q.parent()))
				return false;
			if(parameters.length != q.parameters.length)
				return false;
			for(int i=0;i<parameters.length;++i)
				if(parameters[i] == null ? q.parameters[i] != null : !parameters[i].equals(q.parameters[i]))
					return false;
			return true;
		}

		@Override
		public int hashCode() {
			int result = QueryRuleInstruction.this.hashCode();
			for(Object parameter : parameters) {
				result *= 31;
				if(parameter != null)
					result += parameter.hashCode();
				else
					System.err.println("Parameter is null!!!");
			}
			return result;
		}

		@Override
		public IModification perform(ReadGraph g) throws DatabaseException {			
			final Object[] bindings = new Object[workSpace];
			System.arraycopy(parameters, 0, bindings, 0, parameters.length);
			return rule.execute(g, bindings);			
		}
		
	}
	
	@Override
	public IModification execute(ReadGraph g, Object[] bindings) throws DatabaseException {
		Object[] parameters = new Object[variables.length];
		for(int i=0;i<variables.length;++i)
			parameters[i] = bindings[variables[i]];
		IModification result = g.syncRequest(new Query(parameters));
		return result;
	}

	@Override
	public void doExecute(WriteGraph g, Object[] bindings) throws DatabaseException {
		Object[] parameters = new Object[variables.length];
		for(int i=0;i<variables.length;++i)
			parameters[i] = bindings[variables[i]];
		IModification modi = g.syncRequest(new Query(parameters));
		if(modi != null)
			modi.perform(g);
	}
	
	@Override
	public void mapVariables(TIntIntHashMap map) {
		for(int i=0;i<variables.length;++i)
			variables[i] = map.get(variables[i]);		
	}
	
	@Override
	public void toString(StringBuilder b, int indent) {
		b.append("QUERY[");
		b.append(workSpace);
		b.append("]");
		for(int i=0;i<variables.length;++i)
			b.append(" " + variables[i] + "->" + i);
		b.append('\n');
		for(int i=0;i<indent;++i)
			b.append(INDENTATION);
		rule.toString(b, indent);
	}
	
}
